# -*- coding: utf-8 -*-
#!/usr/bin/env python

# Copyright (C) 2011 Johan Hake
#
# This file is part of DOLFIN.
#
# DOLFIN is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# DOLFIN is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with DOLFIN. If not, see <http://www.gnu.org/licenses/>.

from __future__ import print_function

# Template for script sent to qsub
template = """#!/bin/sh

# Your job name 
#$ -N %(jobname)s

# Resources
#$ -l %(resources)s

# enter the job-directory
cd %(workdir)s

# Use current working directory
#$ -cwd

# Join stdout and stderr
#$ -j y

# Execute in current environment
%(environment)s

# Set queue
#$ -q %(queue)s

# Set your number of processors here. 
# Requests mpich environment although we actually are using openmpi
#$ -pe %(PE)s %(np)d

# add /usr/local/bin in path
PATH=$PATH:/usr/local/bin

# Run job through bash shell
#$ -S /bin/bash

# setup environment
%(paths)s
%(setup)s

%(job)s
"""


def job_script(job, jobname, nodes, ppn, walltime, mem, vmem, workdir, \
               email, paths, setup, keep_environment, queue, parallel_environment):
    """
    Generate a sge specific job script
    """

    # Set default queue
    queue = queue or "all.q"

    # Check environment
    if not parallel_environment:
        raise RuntimeError("expected proper values for 'parallel_environment'. "\
                           "Check available environments using: qconf -spl")

    # Set resources
    resources="cpu=%d:00:00"%walltime
    if mem:
        mem = mem.replace("b", "").upper()
        resources += ",mem_free=" + mem
    if vmem:
        vmem = vmem.replace("b", "").upper()
        resources += ",h_vmem=" + vmem
    
    # Issue warnings on not supported arguments
    if email:
        print("Warning: 'email' is not supported for the 'sge' backend")
    
    args = dict(jobname=jobname,
                resources=resources,
                environment="#$ -V" if keep_environment else "# No",
                queue=queue,
                PE=parallel_environment,
                np = nodes*ppn, # FIXME: Is this the correct way to set this?
                workdir=workdir,
                paths=paths,
                setup=setup,
                job=job)
    return template % args
