#define _GNU_SOURCE

#include "Pidfile.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <sys/types.h>	// pid_t, getpid, kill
#include <sys/signal.h> // kill
#include <unistd.h>		// access, getpid, unlink
#include <errno.h>		// errno
#include <stdlib.h>		// free

using namespace std;
using namespace stringf;

/*
# Create missing directories
for my $d ('cfgdir', 'bindir', 'piddir')
{
	if (! -d $cfg{$d})
	{
		my @parts = split('/', $cfg{$d});
		for (my $i = 1; $i < @parts; $i++)
		{
			my $dir = join('/', @parts[0..$i]);
			if (! -d $dir)
			{
				mkdir $dir, 0755 or die "Could not create directory $dir";
				print STDERR "Created dir ", join('/', @parts[0..$i]), "\n";
			}
		}
	}
}
*/

Pidfile::Pidfile(const string& tag, const string& dir) : delete_on_exit(false)
{
	if (dir.size())
		name = dir + "/" + tag + ".pid";
	else
		name = PID_DIR "/" + tag + ".pid";
}

Pidfile::~Pidfile()
{
	if (delete_on_exit) remove();
}

pid_t Pidfile::read()
	throw (FileException)
{
	pid_t res;
	FILE* in = fopen(name.c_str(), "r");
	if (!in) return 0;
	if (fscanf(in, "%d", &res) != 1)
		return 0;
	fclose(in);
	return res;
}

bool Pidfile::kill(int sig)
	throw (SystemException)
{
	if (!is_active())
		return false;
	pid_t pid = read();
	if (::kill(pid, sig) == -1)
		throw SystemException(errno, "Killing process " + fmt(pid));
	return true;
}

bool Pidfile::is_active()
	throw (FileException)
{
	pid_t pid = read();
	if (!pid) return false;
	char pname[15];
	snprintf(pname, 15, "/proc/%d", pid);
	return access(pname, F_OK) == 0;
}

void Pidfile::takeover() throw (FileException) { takeover(getpid()); }

void Pidfile::takeover(pid_t pid)
	throw (FileException)
{
	FILE* out = fopen(name.c_str(), "w");
	if (!out)
		throw FileException(errno, "opening " + name + " for write, truncating");
	fprintf(out, "%i\n", pid);
	fclose(out);
	delete_on_exit = true;
}

void Pidfile::remove()
	throw (FileException)
{
	if (unlink(name.c_str()) == -1)
		throw FileException(errno, "deleting file " + name);
	delete_on_exit = false;
}

// vim:set ts=4 sw=4:
