/*
 * Copyright (c) 2007, Dennis M. Sosnoski All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that the
 * following conditions are met:
 * 
 * Redistributions of source code must retain the above copyright notice, this list of conditions and the following
 * disclaimer. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the
 * following disclaimer in the documentation and/or other materials provided with the distribution. Neither the name of
 * JiBX nor the names of its contributors may be used to endorse or promote products derived from this software without
 * specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.jibx.binding.generator;

import java.lang.reflect.Modifier;

import org.jibx.binding.model.IClassItem;
import org.jibx.binding.util.StringArray;
import org.jibx.runtime.IUnmarshallingContext;
import org.jibx.runtime.JiBXException;

/**
 * Member field customization information.
 * 
 * @author Dennis M. Sosnoski
 */
public class MemberFieldCustom extends MemberCustom
{
    /** Enumeration of allowed attribute names */
    public static final StringArray s_allowedAttributes =
        new StringArray(new String[] { "field" }, MemberCustom.s_allowedAttributes);
    
    // values specific to member level
    private boolean m_isPrivate; // internal use, not included in binding
    
    private String m_fieldName;
    
    /**
     * Constructor.
     * 
     * @param parent
     */
    protected MemberFieldCustom(NestingBase parent) {
        super(parent);
    }
    
    /**
     * Constructor with name known.
     * 
     * @param parent
     * @param name
     */
    protected MemberFieldCustom(NestingBase parent, String name) {
        super(parent, name);
    }
    
    /**
     * Make sure all attributes are defined.
     * 
     * @param uctx unmarshalling context
     */
    protected void preSet(IUnmarshallingContext uctx) {
        validateAttributes(uctx, s_allowedAttributes);
    }
    
    /**
     * Check if member represents a property.
     * 
     * @return <code>false</code>
     */
    public boolean isProperty() {
        return false;
    }
    
    /**
     * Check if a private member.
     *
     * @return <code>true</code> if private, <code>false</code> if not
     */
    public boolean isPrivate() {
        return m_isPrivate;
    }
    
    /**
     * Check if collection member.
     * 
     * @return <code>false</code>
     */
    public boolean isCollection() {
        return false;
    }
    
    /**
     * Get field name.
     * 
     * @return field name (<code>null</code> if none)
     */
    public String getFieldName() {
        return m_fieldName;
    }
    
    /**
     * Post-set method that converts the field name to a member name.
     * 
     * @throws JiBXException
     */
    protected void postSet() throws JiBXException {
        if (m_fieldName == null) {
            throw new JiBXException("'field' attribute is required for <field> element");
        } else {
            ClassCustom clas = (ClassCustom)getParent();
            setBaseName(memberNameFromField(m_fieldName, clas.getStripPrefixes(), clas.getStripSuffixes()));
        }
    }
    
    /**
     * Complete customization information based on field information.
     * 
     * @param field (<code>null</code> if none available)
     * @param req required member flag (<code>null</code> if unspecified)
     * @param style representation style (<code>null</code> if unspecified)
     */
    /* package */void completeField(IClassItem field, Boolean req, Integer style) {
        if (m_fieldName == null) {
            m_fieldName = field.getName();
        }
        m_isPrivate = Modifier.isPrivate(field.getAccessFlags());
        complete(field == null ? null : field.getTypeName(), req, style);
    }
    
    private static MemberFieldCustom factory(IUnmarshallingContext ictx) {
        return new MemberFieldCustom(getContainingClass(ictx));
    }
}