use nettle_sys::{
    nettle_hash, nettle_sha512_224, nettle_sha512_224_digest,
    nettle_sha512_224_init, nettle_sha512_update, sha512_ctx,
};
use std::default::Default;
use std::mem::zeroed;

use crate::hash::NettleHash;
use crate::hash::Hash;

/// 512 bit variant of the Secure Hash Algorithm 2 (SHA-2) defined in FIPS 180-4, truncated to 224
/// bit.
#[allow(non_camel_case_types)]
pub struct Sha512_224 {
    context: sha512_ctx,
}

impl_write_for_hash!(Sha512_224);

impl Clone for Sha512_224 {
    fn clone(&self) -> Self {
        use std::intrinsics::copy_nonoverlapping;

        unsafe {
            let mut ctx: sha512_ctx = zeroed();
            copy_nonoverlapping(&self.context, &mut ctx, 1);

            Sha512_224 { context: ctx }
        }
    }
}

impl Default for Sha512_224 {
    fn default() -> Self {
        let mut ctx = unsafe { zeroed() };

        unsafe {
            nettle_sha512_224_init(&mut ctx as *mut _);
        }

        Sha512_224 { context: ctx }
    }
}

impl Hash for Sha512_224 {
    fn digest_size(&self) -> usize {
        ::nettle_sys::SHA512_224_DIGEST_SIZE as usize
    }

    fn update(&mut self, data: &[u8]) {
        unsafe {
            nettle_sha512_update(
                &mut self.context as *mut _,
                data.len(),
                data.as_ptr(),
            );
        }
    }

    fn digest(&mut self, digest: &mut [u8]) {
        unsafe {
            nettle_sha512_224_digest(
                &mut self.context as *mut _,
                digest.len(),
                digest.as_mut_ptr(),
            );
        }
    }

    fn box_clone(&self) -> Box<dyn Hash> {
        Box::new(self.clone())
    }
}

impl NettleHash for Sha512_224 {
    type Context = sha512_ctx;

    unsafe fn nettle_hash() -> &'static nettle_hash {
        &nettle_sha512_224
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn do_nothing() {
        let _ = Sha512_224::default();
    }

    //  CAVS 21.1
    //  "SHA-512/224 ShortMsg" information for "sha_values"
    //  SHA-512/224 tests are configured for BYTE oriented implementations
    //  Generated on Mon Jul 10 14:34:13 2017
    #[test]
    fn nist_cavs_short_msg() {
        let mut ctx = Sha512_224::default();
        let mut digest = vec![0u8; ctx.digest_size()];

        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x6e\xd0\xdd\x02\x80\x6f\xa8\x9e\x25\xde\x06\x0c\x19\xd3\xac\x86\xca\xbb\x87\xd6\xa0\xdd\xd0\x5c\x33\x3b\x84\xf4"[..]);

        ctx.update(b"\xcf");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x41\x99\x23\x9e\x87\xd4\x7b\x6f\xed\xa0\x16\x80\x2b\xf3\x67\xfb\x6e\x8b\x56\x55\xef\xf6\x22\x5c\xb2\x66\x8f\x4a"[..]);

        ctx.update(b"\xca\x2d");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x39\x2b\x99\xb5\x93\xb8\x5e\x14\x7f\x03\x19\x86\xc2\xa9\xed\xfd\xb4\xff\xd9\xf2\x4c\x77\xc4\x52\xd3\x39\xc9\xfc"[..]);

        ctx.update(b"\x49\x76\x04");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xa9\xc3\x45\xd5\x8a\x95\x9a\xf2\x0a\x42\xc8\x4e\x28\x52\x3b\xa4\x7e\x3b\xf8\xfa\xd8\xe8\xc3\xf3\x2b\x7a\x72\xae"[..]);

        ctx.update(b"\xc2\xeb\x0c\x88");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xc5\xb9\x7f\x1e\x46\xd9\x1c\x39\x40\xbe\x8c\xfe\xf1\xe4\x9b\xaa\x77\xb4\x75\xdb\x94\x3e\x49\xf0\x0c\xdd\xbe\x72"[..]);

        ctx.update(b"\xd2\x8b\xc2\xaa\x3f");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xe7\x9d\xe5\x52\x12\x0a\x87\x58\x5e\x73\x13\xe2\x4e\xcd\x74\x13\x9a\x3b\xe9\xba\x28\x12\x93\x6c\x4b\x33\xca\x23"[..]);

        ctx.update(b"\xb4\x14\x2c\xa7\x9f\x6d");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x2a\x0e\xd8\xeb\x0e\x18\xbc\x18\x9f\x5c\x83\x6d\xc6\xa5\x82\x5b\xaa\x6f\x72\x86\xfa\x5b\x96\x89\xb6\x41\xbb\x3b"[..]);

        ctx.update(b"\x8d\xae\xdf\x3c\x42\x0c\x8c");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x4e\xda\xc9\x1a\x8e\xba\x35\xa1\x43\x1e\xf4\xd4\x97\x05\xdb\x12\xda\x38\x2a\x78\x6d\x15\x63\x67\x74\xcb\xd7\x08"[..]);

        ctx.update(b"\x69\x63\x44\x69\x13\x77\x14\x10");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x21\xf6\xc3\x73\x63\x7e\x6a\x5e\x89\xd6\xe8\x88\x11\x11\x0c\x5c\x3f\xa1\x2e\x49\x71\x44\x91\x29\x14\xc5\x46\xe1"[..]);

        ctx.update(b"\x6f\x31\x6a\xb2\xb4\x0f\x09\x68\x19");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x58\x99\x54\x2e\x9a\x78\xd1\x40\x39\x29\xc4\x34\x69\x93\xe2\x13\xb8\x3c\x32\xf9\x12\x74\xbd\xdb\xc6\x3e\x41\x92"[..]);

        ctx.update(b"\x58\x19\xa7\x7f\xa1\xa8\xf9\xc3\x5d\x17");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x2e\x94\x4a\x5c\x53\x4b\x80\x92\x2f\x6c\x21\xe9\x72\x77\xb4\xa0\x1a\x4a\x0f\xa6\x52\xd6\xd4\x0b\x37\xfb\x0f\xc9"[..]);

        ctx.update(b"\x3c\x7e\x03\x84\x01\xfa\x74\xc6\xc0\x6e\x41");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xa7\x4a\xf6\x88\x19\xaf\xe8\x1b\xcd\xac\xeb\xa6\x42\x01\xc0\xd4\x1f\x84\x3e\x4b\x08\xe4\x00\x2a\x37\x5b\xe7\x61"[..]);

        ctx.update(b"\xe2\x43\xb9\xf9\x01\x85\x54\x47\xfa\x2a\xf3\x02");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x33\x0c\x77\xed\x20\xdc\x93\xca\xbf\x0d\xf3\x44\x2c\x62\x0c\x9d\xf5\xc7\xa0\x2d\x7c\x79\xfc\x1d\x2f\xf7\xff\x94"[..]);

        ctx.update(b"\xc5\xf9\x42\xed\xc4\x64\x07\xc6\x01\x9c\xd9\x26\xa0");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xe1\x80\xa0\xd8\x36\x83\xcb\x7e\x79\xd8\x12\xab\x5d\x42\x88\x39\x9c\x78\x89\x5c\x6a\x91\x8e\x99\xa2\x83\x91\x55"[..]);

        ctx.update(b"\xe5\x7d\x4c\x47\x8a\xd6\xa6\x4b\x8b\xf3\x35\x2e\x75\xb6");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x18\xb3\xe8\xf5\xa1\xfb\x60\x3c\x67\x06\x0c\x12\xa7\x36\xe5\x7c\x5c\xcc\x0b\x89\xfc\x62\xc9\x0b\x02\xdd\x92\x6b"[..]);

        ctx.update(
            b"\x32\xd0\x49\x92\x8a\x58\x3b\x5b\x95\x47\x85\x80\x5d\x8b\x26",
        );
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x18\xf8\x7e\x8e\x6b\x54\xec\x43\x36\x44\x2f\x76\xed\xa0\xb3\x2d\xcb\x64\xd4\x63\xc0\xbe\x69\xe3\x48\xf8\x6c\x34"[..]);

        ctx.update(
            b"\x3b\x25\x42\xcb\x3f\xa5\x60\xdd\x0d\x79\x6d\x59\xee\x15\x1f\xee",
        );
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xdc\x4a\xb3\x4d\xb2\x41\xb3\xfc\x20\x55\xc5\x4f\x70\xa8\xfd\x30\x45\x23\xd8\xc4\x20\x28\x52\xf5\xf0\x77\x64\x4f"[..]);

        ctx.update(b"\xec\xbe\x36\x79\x59\xa9\x1b\x15\xeb\x08\x04\x8b\x33\x75\x2e\xf7\x6d");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xb7\x6e\xaa\x7d\xde\x45\xed\x50\xc0\x7b\x5b\xaf\xaf\xbf\xdc\xc0\x95\x9c\x1a\x64\xec\xa7\xad\xc3\x91\xab\xa2\x45"[..]);

        ctx.update(b"\x40\xbd\x7d\x47\xb6\x36\xc2\xa7\x49\xa2\x47\xfd\xda\x75\x80\x7c\x23\x8b");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x08\x7e\xd6\x8f\x1d\xb9\x0f\xfb\x2f\xb4\xff\x7d\xc4\xb1\x7f\xe0\x81\x00\xb6\x43\x83\x85\x03\x78\xef\x54\x33\x39"[..]);

        ctx.update(b"\x33\x1c\x89\x39\xef\x68\x5a\xe0\xef\xcd\x40\x30\x37\x8a\x8b\x0a\xb7\x20\xc7");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x97\x87\x4d\x56\x29\x6a\x28\xf8\xdf\x28\x04\x48\xf7\x6a\x9e\x80\x66\x05\xda\x25\x7a\x60\x83\x59\x8f\x23\x2e\x4d"[..]);

        ctx.update(b"\x26\x63\x4d\x1e\xb3\xcc\x27\x93\x83\x0c\x97\xf4\x96\x74\xbd\xa7\x47\x43\xeb\x8f");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x82\x94\xb1\x12\x58\x8d\xfb\x9c\x55\x3a\xc8\x8f\xbd\x84\x41\x72\x2c\xec\x6a\x1d\x43\x53\x11\xfe\x72\xe0\x11\x0c"[..]);

        ctx.update(b"\x62\xed\x23\xab\x88\xc3\x0f\xe1\x35\x0f\x70\x2f\x3b\x6d\x45\x25\x3a\x0f\xc6\xae\xc1");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x30\x9d\xb7\x06\x38\x77\x1e\x2e\xec\x30\x9a\xbf\xda\x21\x9e\x5f\x13\xba\xc5\xfa\x9d\x5e\x84\xe7\x27\x76\x4b\xca"[..]);

        ctx.update(b"\x82\x7d\xbb\xe3\x6f\xb2\x66\x3c\x95\xea\xf8\x1c\x7c\xe4\x41\xdc\xa6\x64\x05\x05\x0a\xd8");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x39\xe7\x86\xa9\x3d\xc9\x0e\x96\x47\x1e\x46\x69\x50\xa5\xa0\xf7\xf1\x42\xca\x69\xc1\x25\x1b\x93\x2e\x8b\xbf\x45"[..]);

        ctx.update(b"\xa2\xad\xa6\x1a\x95\xcf\x8f\xb6\xf9\x3c\xbd\x0a\xa6\x7b\x70\xe2\x86\xdc\x85\x3a\xfa\x51\xf8");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xe8\x21\x24\x7c\x7b\xc7\xac\x8e\xf0\x7f\xb3\x1a\xc2\x5f\xb4\x8a\x3f\xbc\x50\xfd\xe3\x99\xa3\xcb\x58\x89\x54\x5f"[..]);

        ctx.update(b"\x56\x33\xee\x7b\x88\x70\x00\x2c\x01\xa3\xc6\xf6\x8f\x78\x12\xf1\x3a\xdf\xe0\xbf\x6f\xa7\x94\xbf");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xc3\x4b\x46\x78\x58\xb6\x52\x02\xfd\x8c\x73\xff\xac\x67\xcd\x8e\xe5\x51\x18\xd1\x2b\xeb\x2b\x11\x0b\x63\xae\x43"[..]);

        ctx.update(b"\xda\x1d\xd1\xeb\xea\x3c\xc9\x47\x4c\xf2\x55\xdc\x23\xe3\x56\x95\x02\x6e\x86\xd6\x09\x3a\xba\xb2\x51");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xe6\x50\x80\x2f\x29\xad\x20\xb9\x00\x14\x8c\xf3\xe9\x36\x13\x99\xe6\x20\x7d\x45\x52\x57\xfb\xae\x3e\xfe\xe5\x0c"[..]);

        ctx.update(b"\x8f\x71\xc9\x34\x4c\xe6\xa5\x35\xd1\x04\x80\x3d\x22\x11\x9f\x57\xb5\xf9\x47\x7e\x25\x38\x17\xa5\x2a\xfa");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x6b\x3a\x28\x8b\xd8\x41\x37\xcd\x76\x00\xcc\xf2\xd2\xab\x9b\x79\x7d\x0e\x87\xa1\x8c\xcb\x46\x81\x83\x39\x62\x35"[..]);

        ctx.update(b"\xd6\x68\xa2\x17\x80\x59\x40\x24\xdb\x02\x5d\xcb\x37\x3e\xf3\x63\x9b\xcd\x57\x43\x14\x55\x27\xe2\x4d\xa6\x7d");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x2b\x93\x44\x6d\x3a\xa2\xf1\xf9\xea\x85\x8e\x47\x80\x6c\xc4\x72\xf4\x0e\x33\x08\x5b\x7c\x7a\x65\x22\xcf\x65\x48"[..]);

        ctx.update(b"\x03\xf8\xcf\x98\x03\x5e\xac\x15\x8f\xf1\x8f\xd9\x3b\x44\xcd\xb6\x21\x59\x72\x55\xd7\xc3\xd9\x7a\x9e\x08\x79\x89");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xe1\xfc\x5f\x50\x8d\xee\xd8\xbf\x74\x88\x3b\x6f\xde\xf8\xc3\xe0\xc8\x63\x06\xe7\xf1\xd6\x30\x0c\xce\x4d\x36\xdc"[..]);

        ctx.update(b"\x2a\x1f\x57\x99\x83\x60\xe9\x21\x6f\x50\x40\xe8\xf7\x5f\x5b\xff\xba\x43\x68\xee\xed\xed\xe5\x4a\xa0\xbb\x05\x8a\x43");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xbd\xc9\x1e\x67\x9f\xc2\xf4\xe4\x0a\xf1\x3f\x03\xec\xa3\xf6\x1a\x7b\xa3\x3a\xc2\xc6\x13\xd9\x48\xed\x7f\xc0\xbc"[..]);

        ctx.update(b"\xcd\x5f\xee\x5f\xde\x5e\x9a\xa2\x88\x4b\x4f\x48\x82\xcf\xa7\xd5\x57\x1f\x8f\xd5\x72\xc5\xf9\xbf\x77\xa3\xd2\x1f\xda\x35");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x72\xb4\x34\x17\xb0\x71\xf4\x81\x18\x33\x02\x77\x31\xb0\xca\x28\x54\x9c\x03\x57\x53\x0f\xe2\x58\xca\x00\x53\x3e"[..]);

        ctx.update(b"\xa9\xa7\x30\x38\xa2\x8e\x40\xfe\xb3\x74\x81\x9d\xf3\xa2\x6d\xb8\x46\xfd\xa2\x84\xde\x7e\x88\xdb\x01\xe2\x34\x8c\x0c\xa3\x8c");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xa5\xba\xdd\xce\xed\x5b\x58\x61\x4c\x95\xb8\xa6\x21\xd5\x8c\xd4\xb1\x25\x59\xbd\x34\x79\x5e\xf5\xbe\x9c\xd6\x03"[..]);

        ctx.update(b"\xbe\x48\x50\xf5\xbe\x6c\xb6\x8c\xdb\xa5\x94\x8e\x3b\x94\xdb\xbc\xe8\x29\x89\xaa\x75\xb9\x70\x73\xe5\x51\x39\xaa\xc8\x49\xa8\x94");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xc1\x1e\xed\xaa\xdc\x52\x49\xce\xfa\xfc\x5a\x9f\xad\xf3\x9e\x81\x2d\xb7\x00\xbd\x27\x67\xb8\xde\x1c\xdd\x60\x99"[..]);

        ctx.update(b"\xce\xd7\x8b\xf0\x73\x4a\x52\x7f\x5c\x07\xef\x6c\x8a\xdf\x80\xae\x25\x2b\x8c\xff\x10\x0d\x39\x3e\xc1\xf8\xc0\x11\x50\x72\x1f\xce\xcd");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xd4\x23\x7a\xa1\x95\x6d\x37\xa6\xdc\xab\x7b\x07\x26\x9f\x18\x22\x8f\x7e\x31\x56\xc3\x55\x5c\xf4\xd3\xd6\x8c\x1a"[..]);

        ctx.update(b"\xe4\x00\xbc\x89\x94\xcf\xf2\x3e\xae\x18\x9d\xf3\xb1\xe7\xe4\x40\xa6\xb5\xb1\x78\x61\x97\x12\x81\x05\xb5\xdf\xe2\xc6\xfe\x9d\x1e\xf3\x1a");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x45\x35\x1a\xdf\xfe\xfc\xfb\x5e\x39\x08\x54\x61\xf8\xbf\xf5\xf5\x65\x2a\x5a\xa5\xa0\x40\x1f\x04\xd4\xf5\x64\x52"[..]);

        ctx.update(b"\xf1\x52\x84\xa1\x1c\x61\xe1\x29\xea\x06\x06\xbd\x65\x31\xf2\xf1\x21\x37\x76\xe0\x1e\x25\x3d\x1d\xef\x53\x0b\xed\x1c\x3c\x42\xb3\xc6\x8c\xaa");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xfa\x42\x5b\xc7\x32\xd6\x03\x35\x66\xc0\x73\x56\x0b\x2c\x5f\xe3\x22\xaa\x4f\xa2\x2a\xaa\x3e\xc5\x11\x54\xff\xd8"[..]);

        ctx.update(b"\xd8\x58\xee\xd9\x42\xfe\x92\x6b\x52\xba\xda\x62\x3f\xb8\xd0\xf1\x84\x58\xe4\x16\x33\x32\xe1\xc1\x8c\x58\x1f\x38\xce\xda\xfb\xba\x4d\x90\x84\x62");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xb2\xb3\x08\xdb\x94\x8d\x71\x69\xc8\xee\xbb\x4d\x7a\x19\xbe\x96\xd5\xc4\xb5\x84\x95\x93\xec\x58\x7d\x38\x65\xe6"[..]);

        ctx.update(b"\x78\xb8\xad\x23\x72\x63\x23\x98\xcf\x56\x71\x7d\xf3\xb1\x41\xa4\xf3\x8d\x5d\x83\x02\x9b\xc1\xf0\x2d\x07\xf5\x29\x49\xa3\x99\x17\x29\x41\x41\xd4\x14");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x71\xbe\x05\x86\x2b\x44\xe9\xf5\x7a\xd9\xbe\x17\xf1\xde\x18\x54\x71\xea\xbd\x11\x1c\x17\x25\xe8\x35\xe0\x96\x25"[..]);

        ctx.update(b"\xfa\xff\x89\x4c\x4a\x2e\xd7\x0c\x4c\x9d\x67\x03\xb6\xbc\x3a\x3d\x2b\x2c\xba\x8f\x04\x76\x9d\xe3\x50\xfc\x0c\xd0\x19\x30\x05\x6e\xea\x03\xfe\x68\x8d\xa1");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x59\xf5\xa8\xc8\x43\xb9\x82\x98\x9a\x08\xe6\x85\x2e\x34\xf0\xc0\x33\x97\xc9\x9a\x87\x86\x4d\xfb\x1b\xba\xfe\x1c"[..]);

        ctx.update(b"\xe6\x06\x5e\x1e\xb7\xcf\x90\x1c\x3b\x08\xdb\xa4\xae\xca\x7d\x46\x73\xc3\xfd\xbe\x60\x39\x1c\x1b\x5d\x4d\x09\x20\x2d\x7d\xd4\xea\x8b\x6d\x25\xd3\x79\xe7\x2e");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xea\x85\x64\xc9\x88\xcb\x67\xe3\x02\xbd\x40\xdf\x21\xa5\xf7\x96\xa0\xa1\x16\x84\x64\x93\xc8\x6c\x3a\x1d\x08\x0e"[..]);

        ctx.update(b"\x02\x9d\x14\xc5\xa4\xa5\x7a\x64\x9a\x86\xc5\xf2\xe0\x22\xe6\xee\x09\x8f\x29\xc9\xd4\x60\x65\x48\xff\x27\x75\x29\x60\x38\xec\x09\x31\xaf\x05\x85\xac\xb4\xd9\xa0");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x71\x66\x68\xcb\xa7\xb1\xc0\xdd\xc1\xc4\xb3\xe6\xfe\x3f\x89\x17\xea\xda\x62\x50\x6b\x68\x55\xe5\xaf\x2b\x54\xe5"[..]);

        ctx.update(b"\xe3\x08\x9c\x05\xce\x15\x49\xc4\x7a\x97\x78\x5d\x82\x47\x4c\x73\xcb\x09\x6a\x27\xc5\x20\x5d\xe8\xed\x9e\x3a\x8c\x97\x1f\x7f\xa0\xea\xb7\x41\xfd\x2c\x29\x87\x9c\x40");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x75\xa0\xfa\x97\x8c\x45\xd2\x68\x12\x4d\x8c\xd9\xef\x0a\x08\xec\xab\xbb\xed\x53\x41\x2c\xfc\x7c\xb1\xc0\x03\x98"[..]);

        ctx.update(b"\x90\x1d\xb0\xf7\x0f\x2d\x10\x62\x6a\x8a\xf4\x08\xc5\xd1\x51\x96\x69\x3d\x49\x38\x21\x78\xad\x0d\xe3\xb1\xc1\xd8\xb3\xef\x3f\xf8\xc5\xa5\x9e\x9a\xe2\x86\x93\x2d\x0c\x9a");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x14\x70\xa4\x19\x9f\xe4\xc7\xbc\x33\x07\x60\xc1\x19\x7e\x79\x45\x7b\x16\x44\x76\x4e\x28\xa6\x2d\x38\x6a\x78\xb4"[..]);

        ctx.update(b"\x04\x31\xa7\xbf\xbb\xec\x1b\xb8\x11\x6a\x62\xe1\xdb\x7e\x13\x46\x86\x2d\x31\xad\x51\x10\xff\x1b\xb9\xfa\x16\x9a\x35\xdb\xb4\x3a\x24\xe4\x57\x56\x04\xec\x8b\x18\xe4\x13\x00");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x1e\x75\x90\xe4\x08\xc0\x38\xb7\x94\xe9\x82\x0b\x25\xd0\x11\xc2\x62\x06\x2b\x96\xd1\x11\xdc\xcc\x46\xdc\x67\x83"[..]);

        ctx.update(b"\xd1\x41\x75\x3c\xef\x2e\x1b\x44\x2e\xb0\x17\x08\xf4\x67\x19\xcd\x86\x53\xb2\x2e\x1d\x97\x09\x07\x3d\x7c\xcf\x92\x42\xb8\x75\xd0\x3c\xf1\x9c\x2b\xfc\x7d\xcd\x59\x6f\x6c\x6a\x83");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x0f\x35\x7b\x8c\xa6\x76\xa8\x6c\xa6\x36\x20\xb2\x23\x93\x27\xac\xcb\xd4\x25\x28\x14\xa6\xb8\xc5\x68\x95\xea\x58"[..]);

        ctx.update(b"\xce\x94\xe0\x12\x6b\x02\x0d\x68\x11\x5a\x5e\xa4\x10\x83\xb4\x71\x7a\x13\xd9\xc5\x7a\x5b\x6d\x0a\xe7\xa3\x23\x7c\x19\xb6\x2d\xd2\xfc\xa2\xf4\x20\x62\xb9\x0d\xfc\x9e\xdb\x68\xe0\x4e");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x93\x5d\x39\x1d\x62\x6a\x13\x57\x72\x31\x9e\xc9\x3f\xd6\x52\xbd\x9f\xd5\x68\x01\x07\xc9\x87\x87\x2a\xe6\x72\x12"[..]);

        ctx.update(b"\xef\xc2\xd8\x1b\x30\x04\x3b\xb2\x6c\x8d\x2c\x7a\x7d\x5b\xfe\x8e\xe5\x79\xe4\x63\x12\x2d\x56\xa0\xe1\x72\xb6\x30\x60\xa8\x34\x95\xd8\x8b\xef\x7f\x0b\x62\x5c\x9d\xba\x78\xec\xb3\x6a\xd0");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xd9\x21\x47\x89\xb7\xb7\x36\xa7\xb2\x8c\xeb\xff\x88\x29\xa3\x99\x9f\xf0\xc7\x49\x60\x95\xb1\x30\xc1\xe2\xe8\x29"[..]);

        ctx.update(b"\x5b\x27\xd6\xfe\x99\x81\xd1\x36\xb5\x21\x57\xb8\x39\xbb\x5b\x78\xd7\x84\x4d\xf5\xa3\x97\xef\xbe\x80\x98\x64\x35\xd8\x98\x89\x93\x91\x39\x89\x2c\x3c\x13\xac\xf5\x8f\x56\x70\x43\x15\xb0\xdb");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x4c\x79\x2b\xb6\xb1\xe3\x3a\x3a\x04\x2d\xd4\xa6\x9a\xda\x80\x2e\x3e\xc0\xbd\x83\x9d\xaa\x88\x63\x5b\x22\x5e\xfb"[..]);

        ctx.update(b"\x4d\xbe\x12\x90\x52\x4b\xd7\x3d\x9d\xb5\xf2\x1f\x9d\x03\x5e\x18\x3d\xc2\x85\xb8\x5b\xa7\x55\x05\x7c\x76\x97\x77\xbe\x22\x7c\x47\x0e\x36\x79\xea\x9a\x73\x55\xd8\x89\xbb\x81\x91\xea\x2e\xa7\xe2");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x33\xa5\xa8\xd6\x11\x9b\xb6\xdd\x7b\x2e\x72\xec\xe8\xe4\xd5\xd0\x2a\xa9\x90\x48\xc0\x45\x91\x69\xee\x9e\x6d\x04"[..]);

        ctx.update(b"\xbd\xfa\x17\x39\x95\xa5\xb9\x9f\xba\x74\xd8\xe2\x55\xb7\x11\x77\x17\x31\xa8\xe1\x85\x1d\x7d\xd6\x93\x73\x3b\x9d\x75\x67\xd3\xba\xb2\x32\x29\xcd\x7a\x16\x0c\xb8\x39\x9c\x6c\x00\x6c\x21\xc7\x76\xb4");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x33\xc9\x34\xb5\x10\x20\x63\x21\x3e\x7b\x53\x67\x5c\x47\xb1\x9c\x39\x7f\xc3\xa5\x1f\xb3\x7a\xb4\x6e\x17\x9b\x10"[..]);

        ctx.update(b"\x94\x63\xc5\x63\xaa\x1f\xda\x89\x47\x57\x39\x8e\x11\xb0\x4a\x4b\x4c\xee\x0d\xce\x9d\xab\x01\xad\x13\x4f\x44\x9a\x67\x3c\xfb\x38\x83\x49\xf4\x2e\x42\x72\x8a\x79\x77\xce\xc9\x06\x30\xc1\xd3\x80\x3c\x72");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x4c\xae\x18\xa3\x9a\x5c\xa4\x51\x2b\x95\x9d\xa0\x6b\xc9\xd6\x3d\x07\x16\x80\x84\xe3\x5a\xaf\x03\xc8\xba\x5d\x8d"[..]);

        ctx.update(b"\x2e\xe6\xe8\x13\xec\x84\xc2\xad\xed\x58\x00\xfc\x1c\xae\x94\xc0\x90\x54\xe2\x5d\x6b\xf0\xc4\xd1\x02\x96\xd1\xe8\x3e\x9a\xdf\xb3\x42\xa3\xc6\x36\x4d\x01\x67\x61\xe2\x41\xed\x42\x27\xfe\x53\x78\x8c\xfa\x27");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x89\xc6\x92\x9b\x3c\x19\xa1\x47\x2a\x45\x24\xe1\x42\x78\x23\x76\x8e\x46\x45\x1e\x48\xfa\xa7\xa7\xdb\x7c\x86\x04"[..]);

        ctx.update(b"\x86\x48\x33\xcb\x29\x12\xfa\x44\x92\x92\x87\xbd\x0d\xb7\x98\x49\x68\xe5\x92\x0d\x49\xf9\x72\x5d\xc4\x87\x07\x22\x2d\x65\x5f\x7f\x2b\x6e\x7f\xb4\x0e\x80\x56\x42\x63\xc7\xf4\x50\xc5\x3e\xf8\x4d\xf6\x72\x47\xd7");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x2c\x10\x2a\x83\x49\xc4\xe3\x62\xab\xd6\xa2\x58\x21\x77\x63\x22\x98\x98\xe2\x7c\x2e\x72\x1c\x73\xbc\x88\x07\x52"[..]);

        ctx.update(b"\xe4\xdf\xe6\x97\x23\xd4\x30\x83\x4b\xdc\xda\x7f\xa1\x38\xf1\xfd\x26\xf2\x26\x74\xbe\x71\x16\x9d\x91\x4a\xcf\xe2\xb2\x59\x90\x14\xff\x11\xd1\x5c\x48\x10\x83\x74\xb3\xf6\x99\x16\x5e\x78\xc7\xb5\x47\xb0\x8a\x6c\x69");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x52\x31\x0b\xac\xf7\x8d\xdd\x40\xd7\xe9\x3e\xb3\x21\xda\x4d\xe5\x41\xa7\x29\x6f\x98\xf0\xad\x97\x45\x91\x58\x87"[..]);

        ctx.update(b"\x6a\x30\x00\x17\x6d\x0b\x02\x99\x92\xd9\x21\xe1\x02\xee\xee\x94\xdf\xb9\xb0\x29\xd4\x01\x9e\x4a\xf0\x6f\x51\x7d\xb4\x4d\x78\x03\x6a\x8d\xce\x00\xb9\x0d\x49\xdd\x77\x19\x94\xeb\xbe\x32\xc0\x49\x42\x0d\x37\x07\xe1\xdf");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x07\xde\x1e\xd3\xf8\xea\x77\x4a\x8e\x5e\x08\x06\x69\x20\x6c\x85\x72\xed\xbc\x3f\x16\x58\x70\x91\x3a\xb1\xe4\xf6"[..]);

        ctx.update(b"\x55\xca\xb8\x76\x55\x60\x79\xef\xf9\x8b\xf7\xb9\x0e\x31\x8f\x8f\xf5\x83\xc2\xbe\x55\xde\x88\x2c\x0d\xef\xe6\x99\x6d\x1b\xc2\x25\xa5\x1e\xf7\x12\x7d\xf2\xa5\xcc\x47\xf2\xca\x26\x12\x3f\x17\xe7\x21\x63\xfc\x85\x9c\x34\x06");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x81\x1f\xd9\x07\xf3\xf7\x4b\x9f\x21\x41\x93\x04\x8d\x70\xed\xa4\x4f\x3a\x74\x65\xcd\xb8\x82\x17\x3b\xf7\x9a\x0a"[..]);

        ctx.update(b"\x5d\xab\x01\x6d\x4f\xeb\x42\xea\x60\x61\x23\xc0\xac\x4c\x32\x59\xcc\x71\x77\xaf\x9c\xc9\x55\xa6\x41\x77\xb2\x9f\x6b\xfc\x58\x4a\xd8\xae\x9a\xbb\x1a\x67\x3a\xc1\x99\x7a\x8a\x73\xdb\xeb\xd2\x0b\x03\xdf\xf2\xf8\x98\x51\x48\x2e");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x34\x0b\x5a\x71\xed\x65\x40\x7a\x68\x67\x06\xb0\x68\xed\xb9\x2d\x0a\xd2\xa0\x79\xc1\xc8\x8b\xf7\x8a\xd8\xa3\xa0"[..]);

        ctx.update(b"\x09\x21\xaf\xb4\x03\xaf\xd3\x1e\x8b\x08\x39\xdc\xc1\x5b\xa2\xc0\xfc\x32\xbc\xc9\x0f\xa7\x8f\xf8\xcc\xd5\x19\x60\xc7\x8e\x35\x7d\x5d\x4c\xc2\xc6\xf4\x69\x00\x85\x83\x92\x3d\xe7\x56\x79\xb2\x4f\x0b\xca\xa8\xf0\xe1\x61\xdf\x9b\x32");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xdd\x30\x88\x2c\xda\x21\x44\xb3\x1a\xb1\x79\xe8\xf9\x05\x46\x01\x68\xb8\x75\x8d\x8b\xe8\xd8\x07\x8d\xd6\x10\xab"[..]);

        ctx.update(b"\xc0\xfa\xcb\x87\xb9\x28\x3d\x87\xfd\xd0\x27\x60\xe4\xfa\x43\x7b\x89\x6b\x13\xde\x28\x78\x6b\xa0\xcb\x19\x2f\x9f\x26\x3e\x5e\x75\xbf\x66\x4f\xa7\x52\x6d\x3b\x20\x73\xa5\xfc\xde\x75\x29\x14\xec\x8b\x6e\xf1\x90\x2d\x1b\x70\xc5\xbd\x1b");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x1e\x7a\xdf\x07\x37\x57\x2d\x1a\xaa\x5d\x4c\x91\x7c\x94\xd0\x5d\xc0\x92\xd6\xe5\x90\x59\xdf\x65\x94\x90\xf1\xc7"[..]);

        ctx.update(b"\xe0\xea\xfe\x1a\xed\x71\x9f\xd7\x12\x2a\x3e\xdc\x9d\xf7\xc4\xbe\xdf\xbc\x1b\x0c\x3b\x1b\xe6\x31\x7c\x82\x6f\x9c\x6b\x68\xe0\x1e\xa2\xd3\xb7\x93\x9b\x1e\x74\xc4\x0c\xf9\x94\x8c\x77\xb1\xc3\x3a\xfc\xb2\x4e\x37\x73\x4a\xd7\xe9\xb7\xaf\x4c");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xc2\x60\x13\x01\x97\xdc\x77\x74\x31\x3c\x75\x84\x3d\xd8\x20\xd5\x89\xa6\x8d\xf5\x9e\x1a\xef\x7c\xf3\x47\xf5\x65"[..]);

        ctx.update(b"\xc3\x2a\xd1\xf8\x21\xfb\xd8\xb5\xb7\x15\x3e\xa8\xaa\x11\x05\x5e\xa8\xfb\xde\x54\x86\x60\xec\xae\x28\xa8\xcb\x2b\xa2\xa6\xd1\x59\x9c\x6e\x93\x05\xfb\x67\xab\xf6\xbe\xa1\xb9\xdb\x25\x68\xa8\x81\x19\x19\x33\x49\xc8\x34\xa7\xda\x90\xb3\xea\x13");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xc9\x1e\xe4\xc7\x40\xa5\x66\x3b\x3b\x19\xed\x77\x8a\x05\x16\xc0\x1a\x3c\x0c\xbd\x3d\xfd\x6c\x09\xef\x1a\xfd\xb9"[..]);

        ctx.update(b"\x26\x57\xc6\x25\xf5\xea\x51\xf6\x6b\x13\x75\xf4\x79\x12\x3d\x86\x3b\xa0\x51\x4d\x20\x84\x30\x57\x4c\x61\xba\x96\x3a\xd2\xf1\x91\xa6\xb8\x00\x5a\xaa\xd3\x7d\xc0\x27\x90\xeb\xc5\x66\x22\x83\x53\xff\xd4\x34\x2a\xfa\x9e\xe8\x1d\x35\xf6\x09\x7e\x05");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x92\xf1\x33\x4f\x25\x32\x60\xe5\x13\xd5\xe6\xf0\xd1\xa6\xaf\x6e\xd2\xfb\xbc\xdb\x7f\xac\x85\xc3\x9d\x6d\x40\x40"[..]);

        ctx.update(b"\x56\x55\xc3\x67\x1f\xbd\x82\x88\xea\x14\x94\x4a\xf9\x25\xea\xec\x65\x34\x08\xf5\x62\x33\x65\x0f\xc4\x39\x10\x18\xc2\x2f\x99\x77\xfa\xcd\x2f\xac\xb6\xc5\x06\x2b\x6e\x3d\xfc\x6b\x8e\x83\x7b\x3a\x78\xfc\xa1\x53\xcd\x5e\x9e\x32\xe9\x1f\x5e\x27\xac\x2e");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x58\xd6\xe8\x3b\xa0\x28\x33\xba\x0e\x23\xfe\x34\x72\xd2\x8e\x8f\x98\x63\x98\x62\x0f\x74\x64\xff\x18\xe9\xf6\xea"[..]);

        ctx.update(b"\x35\x0b\xd6\x01\x8c\xbb\xed\x86\xe0\x50\xb3\xe7\xe6\x06\x83\xe1\xf6\x19\xb7\x3b\x4c\x2e\x76\x4b\x74\xee\xcd\xbb\x00\xf4\x01\x8d\x7c\xe9\x24\x5d\xa4\xed\xe7\xe6\xd4\x39\xaf\xc8\xbf\x03\xeb\x89\x64\xa2\xa4\xd9\x3e\xf3\x55\x1a\xbb\xfb\xfe\x1f\xf0\xce\x74");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xc8\xec\x85\x4e\x63\x2b\x2c\x13\x2d\x50\x5a\xff\xe1\xbc\x00\xdd\xbe\x99\x45\x2e\x5b\x59\xba\x76\xa0\x2d\xd8\x38"[..]);

        ctx.update(b"\x4c\xd2\x73\x24\xc2\x83\x64\x87\x3c\x6d\xdb\xc3\xe3\xa7\xe2\xcd\xa9\xe8\xa7\x2a\xd2\xf7\x22\x01\xb2\x62\xf8\x74\xb8\x73\x9f\x30\xab\x60\xc3\x43\x34\xc2\xe9\x2f\x9d\x48\x53\x3c\xd8\xad\x23\x12\xc3\xe7\xc3\x86\xaa\xa2\x83\xb5\x0d\xec\x84\x4f\xa4\x32\xd6\x36");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xce\xd6\x08\x17\x61\xff\x52\x59\xf1\x32\xaa\x83\x1b\x7a\x1b\x43\x2d\x09\x3f\xc8\x57\xda\x0e\xee\xb8\x2b\xe7\x1f"[..]);

        ctx.update(b"\xa4\xf2\x06\x3f\xe6\xec\x17\x55\x0d\x7a\xa8\x31\x09\x19\x52\xdc\x4d\xc0\x41\xa2\x83\xc4\x88\x18\x7e\xf9\xb7\x5e\x70\x1a\xb0\xa2\x5d\x6a\xb6\xe5\xcf\x9c\xc7\x02\xcc\xf0\x2c\xec\x05\xa0\x4e\x37\x50\x7a\xcb\xda\x58\xce\xc9\x33\x93\x8a\x8b\x4b\x75\xa4\x42\x5c\xe4");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xaf\x6b\x7f\xaf\xcb\xe9\xcb\x25\x00\x3c\x57\x02\x99\x5f\xd6\x56\x36\x5a\x20\xe1\x13\xc8\xe5\xb1\x03\x12\xcd\xf0"[..]);

        ctx.update(b"\xc1\xd3\x0f\x33\xab\x4b\x41\x0d\x0f\x8a\x80\x9e\xb5\xcb\x68\x1e\xc9\x15\x3b\xde\x26\x69\x3b\x20\xd4\x40\x57\x2c\xce\x35\x0a\xa0\x09\xfe\x81\x08\xb3\xa0\xfc\xae\xb8\xe3\x76\xb4\x8b\x04\x88\x5f\x23\xab\x8b\xb6\x44\x1c\xfd\x4c\xff\xb1\x25\x5a\x7c\x88\xfa\x10\xf7\x79");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x40\x8f\x4f\xae\x70\xd0\xfd\x74\x5a\xe4\x4e\xd6\x67\x40\xbd\xf1\xfe\xc0\xdd\x4e\x23\xdb\x06\x92\x0c\x6f\x8e\xc5"[..]);

        ctx.update(b"\x61\xb3\x4b\x8a\xaa\xe2\xe1\x9e\x2d\x61\x92\x46\x42\x35\x1a\xde\x2b\xaa\x69\x25\xf7\xa3\xa7\x35\xa7\x2e\x76\xb9\x90\x5b\x77\x5a\x00\x4f\x0f\x7a\x24\x94\x6b\x25\x29\xb3\xc1\x44\xa2\x05\xd9\x14\x2a\xc6\xcd\xd7\x4a\x6b\x0d\xca\x9d\x35\xa5\x56\x31\x07\xc3\x8a\x31\x75\x73");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x19\x65\x8b\x28\x5d\xb6\xcb\x9d\xbb\x4c\xea\xbf\xe6\x71\x43\x4c\xf1\xb8\x54\x55\xa8\xb4\xda\x38\x5e\xad\x83\x0c"[..]);

        ctx.update(b"\x34\x13\xb0\x9d\x3d\x12\xa3\x33\x68\x0f\x79\x8c\x0e\x48\x87\x76\x57\x9d\x3f\x6a\x65\xf2\x04\x12\xce\xd5\x3c\x08\x50\x60\xa6\xe1\xc0\x61\xed\x6a\x44\xb9\x7e\x31\xc2\x02\x97\x47\x98\x75\xa1\x45\x8e\x0a\xc0\x8d\x08\xe0\xa8\xd2\x81\xe0\xb2\xdb\x34\x9e\x2f\x83\x1c\x7f\xfc\x05");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x00\x8a\x3c\x6c\xa5\x78\x53\xad\xb1\xaf\x5d\x44\x59\x5e\x62\x01\x6b\x7a\xf4\x72\x26\x49\x2b\x9f\x3c\x1a\x97\x13"[..]);

        ctx.update(b"\xb6\xe6\x90\xd1\x7d\x9d\x7f\x07\x18\x55\x02\xbf\x6d\x27\x5c\x84\xe3\xac\x4f\x5f\x77\xc3\xd4\xb3\x0d\x8e\x10\x66\x03\xbe\x84\x41\x0c\x11\x84\x9a\x3c\x18\xea\x30\x5f\x3e\x0b\x56\x8f\x06\xea\x54\x06\x94\xac\x0a\x30\x6f\x4c\xd6\x04\x85\x90\xb8\x95\x70\x3c\xb4\xe4\xf7\x85\x78\xf9");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xf5\x1d\xd5\x94\xc7\xdf\xec\xce\xf9\x5e\x3d\x40\x24\xbb\x00\xf9\x31\xa9\xe9\x51\x69\x58\xc3\x2d\x22\x59\xf5\xca"[..]);

        ctx.update(b"\x13\xe6\xb1\xb4\xf0\x21\xd6\x10\xc8\x1c\x97\xf0\xf9\x52\xda\xba\x27\x66\x03\x4d\x81\x5b\x5d\xda\x46\x03\xbc\xf7\x88\xba\x60\xee\x31\x54\x1d\x5b\x43\x53\xb9\xf6\x64\x5d\x96\xad\x99\xee\x90\xf6\x52\x4b\x29\x63\xa7\xb7\xe4\x76\xe1\xe8\xee\xb8\x3c\xbc\x03\x05\xeb\x29\x90\x2a\x5d\x72");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x09\x90\x0c\x5a\xe3\x07\x4f\xe7\x3e\x6c\x4e\xef\x51\xf7\x85\xe5\x79\x47\xba\xfb\xe1\xd8\xde\xa3\x88\x68\xe3\xd1"[..]);

        ctx.update(b"\x3a\x30\x0f\x9c\x2d\xc3\x53\x19\x17\x22\x20\x34\x79\x18\x24\xb4\x94\x36\xbe\x4d\xda\x76\x0f\x67\xa2\x1e\x54\x21\x1f\xe7\x73\x94\x93\x5f\xd3\x5c\xc0\x8c\x71\x55\x9f\xd2\x31\xa2\x55\x16\x72\xf4\x63\x24\xe4\x08\x1d\x4e\x9e\x72\x5d\x56\x6b\x9a\x78\x8f\x65\x04\x71\x3e\xc6\xd4\x6d\x46\xfa");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xd7\x1c\x18\xaf\xd4\x48\x11\x3a\x45\xd0\x6b\xa6\x16\xa1\x94\x5a\x56\x7e\x26\xf3\x36\xcb\x94\xa4\x5c\x25\x19\xd7"[..]);

        ctx.update(b"\x81\xc6\x4d\x23\x13\xdc\xe3\x12\x6b\xa1\x6d\x32\x16\xc0\xf3\x34\x91\x0f\x06\x3b\x00\x72\x0b\xe3\x22\xfd\xd6\x82\x85\x58\x7b\xae\xf6\x68\x60\xff\x3c\x2b\x97\x15\x06\xe4\x6b\xdd\x21\x79\x39\x11\xee\x19\x82\xfe\x14\x21\x3d\x8a\x72\x41\x72\x75\x58\xdb\x04\xdf\xf6\xab\xcf\x0b\x14\xad\xf2\x62");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xc8\x5a\xcd\x75\xec\x4d\xde\xf1\xf1\x79\xf9\x8f\xdc\xf8\xe0\x06\xa1\xd8\x0a\xfc\x14\x74\x93\x10\xaa\x97\x59\x9d"[..]);

        ctx.update(b"\x09\x64\x5b\x92\x0b\xf8\x17\x35\xd6\xb5\xca\xa4\x3b\xb7\x6d\xfd\xa7\x49\x70\x03\xa6\x59\x5c\x2c\xa6\x2a\xb7\xf5\xfe\x49\x28\x79\x2e\xc2\xdb\x42\x68\x93\x27\x5f\x50\x61\x6e\xb6\x01\x6a\x45\x9c\x50\xbb\xd5\xa3\x54\xa3\x81\xb3\x26\x67\xca\x5d\xe5\x37\x6f\x6b\xfd\x61\xd8\xbc\xa9\xc5\xb4\x4f\xb7");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x65\x0e\x31\xa8\x1d\xcc\x7f\x34\x98\x90\x0d\x20\x5d\x61\x1b\x10\x1e\x89\xd7\x24\x3e\x1d\xd7\x75\x51\xb6\xc4\x22"[..]);

        ctx.update(b"\x3f\xd1\xd8\xfb\x9e\x7f\xaf\x18\xa4\x9a\x1b\xdc\x77\xfb\xa0\x93\xcb\xe0\x9a\xf0\x0d\x46\x19\x99\xe1\x9a\xee\xe7\xcf\x91\x06\xe6\x39\x55\xf7\xa4\x91\xe1\xd9\xa8\xc9\xae\x57\xc0\x0a\x8f\x05\x00\xa8\xd9\xf5\x07\x56\xc8\x85\x84\x96\x1e\x7a\xf9\x74\x0d\x53\xbf\x74\x57\xd7\x4b\xc9\xb4\xf1\xaf\xc7\x74");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xfb\x25\x89\xe9\x42\x89\x0f\xed\x8f\x99\xec\xfe\x02\xd3\xa5\x16\x83\xa5\xda\x58\x2b\xe9\xda\xdd\x25\x30\x5d\x33"[..]);

        ctx.update(b"\x5a\xf0\xeb\x31\x6e\x79\x3f\xa9\xea\x9d\x51\x41\xfa\x26\xcd\xc3\xcf\xcf\x98\x52\x97\x61\x63\xa8\xac\x1d\x81\xa1\x0b\x5a\xf5\xfd\x09\x6f\xd0\x61\x41\xd3\xba\xa0\xd1\x83\xce\x39\xc6\xa4\xab\x21\xd4\x6a\xc7\xc2\x4b\x38\x07\x43\x46\x09\x8d\x0a\x25\x0e\xd2\x7f\x3e\x98\xf4\x13\xd0\x39\x51\x82\x38\x26\x54");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x8f\x4c\x6a\x61\x32\x1c\x6e\x50\x0c\xf0\xdc\x2b\x7d\x7d\x97\xda\x93\x01\xf7\x14\x20\xfa\xc6\x04\xfe\xae\x19\xf2"[..]);

        ctx.update(b"\xb2\x6d\x32\x33\xb5\x51\x02\x90\x77\xbe\xea\x92\x85\x98\x89\x69\x1a\xc3\x85\x3a\x9f\x97\x2b\x2a\x6e\x8e\x55\xb8\xdb\x40\x91\x4f\x75\x92\x7f\x1b\x89\x2a\x7b\x1d\xe8\x26\x7c\x7a\xea\x0d\x3c\x19\xb3\x05\x9a\x6c\xb5\x47\x3c\x8e\x76\xd1\x78\x8f\x94\x5a\x75\x0c\x06\x8b\x8d\x02\xa4\x6c\xf6\xe3\xea\x1d\xa2\xea");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x4d\x83\x60\xf3\x69\x2d\xef\xa1\x1e\xda\x02\x7f\x94\x09\x9b\x30\x54\xb8\x41\xf8\x09\xd2\xae\xef\xc8\x85\xd2\xb8"[..]);

        ctx.update(b"\x6f\xb0\xf3\x85\x6d\xcc\x9a\xdd\xbd\x15\x59\x16\xf0\x22\x76\x30\x95\x9f\x01\xc7\x64\xac\xab\x99\xbc\x80\xdc\x43\x50\x52\xff\x4b\x1d\x79\x21\x63\x9f\xd2\xba\xec\x44\x62\xca\xba\x21\x3c\x28\x17\xe3\xa5\x17\x0a\x09\x77\x1b\xd9\x51\xc3\x0b\xcc\x70\xba\x66\x34\x06\xce\xea\xac\xd1\x14\x63\x9f\x21\xed\xc2\x0a\x8c");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x71\xf8\xe0\x9f\x91\x1d\xc8\xac\x67\x99\x9c\x10\xc0\x59\xc2\x03\xaf\x4a\xf5\xe4\x36\xee\xd8\xf8\xa9\x7a\x6f\x2d"[..]);

        ctx.update(b"\x84\x71\xe9\x39\xd4\x53\x14\x86\xc1\x4c\x25\x9a\xee\xc6\xf3\xc0\x0d\xfd\xd6\xc0\x50\xa8\xba\xa8\x20\xdb\x71\xcc\x12\x2c\x4e\x0c\x17\x15\xef\x55\xf3\x99\x5a\x6b\xf0\x2a\x4c\xe5\xd3\xcb\xcd\x79\x0f\x72\x08\x79\x39\xb0\x5a\x9e\xbc\x93\xa9\x05\xce\xcc\xf7\x77\xa2\x0d\x36\xed\xdf\xb5\x5e\x53\x52\xaf\xbe\x8d\xdf\x97");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x06\x5c\xb9\x3b\xb2\x40\x60\x03\x77\x64\x40\xd9\xa0\x2a\xf1\xe4\x17\x0e\x77\x7c\x7d\x2c\x13\xa1\x13\x26\x2c\xe0"[..]);

        ctx.update(b"\xc8\x7a\x74\x48\xb7\x18\x9a\x11\xe5\x32\xa4\x32\x08\x74\x18\x64\x07\xfb\x32\x47\x0d\x18\x90\x4c\xdd\x51\x2f\xd2\x65\xa9\x96\x8f\x95\x22\x51\x32\x71\x7f\xa1\x46\x65\x4e\x72\x5a\xd9\x26\x8d\x5f\x06\x2e\x0f\x51\x08\xde\x1a\x1a\x34\x0a\xca\xb3\xab\x1c\x6b\x8c\x2f\xa1\xe9\x2e\x36\x07\x87\x1f\x3d\xa4\xd4\x05\x5f\xfb\xdc");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xa3\x9c\x7c\x5e\xac\xdb\x22\x66\x95\x87\xa1\x84\x6c\x29\x75\xb4\x1f\x9b\x74\x83\x3d\x99\x79\xde\x2d\x7f\x11\xc2"[..]);

        ctx.update(b"\x26\x46\x5e\xa0\x6d\xbb\x61\xb4\x19\x3a\x98\x2c\x98\x4a\x03\x58\x08\x5f\xd6\xf1\x4c\xa0\x27\xff\x60\xed\xe9\xfa\xe9\x21\x1d\x30\x11\x94\x47\x66\x91\xe5\x1f\xc2\x0f\xf6\x8a\xe9\xab\x37\xc8\x64\xd5\x28\xef\x48\x61\x19\x98\x16\xdd\x34\xd8\xe0\x22\xbe\x72\xad\x29\x9a\x4a\x1f\x20\x0f\x1d\x53\xbd\xcb\x70\x09\xd3\x27\xc6\x05");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x21\xaf\xd5\xf9\x23\xbc\x7b\xae\x23\x4f\x14\xff\xd6\x62\xf1\x74\xb4\xd1\x79\xc7\x24\x1d\xdb\x57\x57\x6f\xe5\x02"[..]);

        ctx.update(b"\x5d\xbf\x5c\x27\x11\xe2\x6f\xba\x34\x23\xec\xbf\xf0\xd5\x6e\xb8\xc9\x34\xcb\x6f\x78\xee\x8e\x96\xf6\xe2\xa8\xaa\xd7\x27\x62\x1d\xd9\x50\x96\xb3\xbd\xee\x1f\x2d\x55\x57\x8e\x25\xdf\xa6\x1e\x39\x91\x03\xeb\x6b\x75\x25\x5e\xc8\x77\x14\xec\x48\xbd\x34\x00\xf6\x36\x7f\x35\x50\x76\x12\xc9\xf1\xc1\xeb\x4f\x6d\x65\xf1\xa3\x7d\x4d");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xec\x5a\x13\x99\x36\xf2\xc4\x2a\x1f\x41\x10\xe2\xc5\x2b\xf7\xc3\xcd\x3b\xae\x49\x95\x5c\x60\x8f\xad\xfa\x16\x3d"[..]);

        ctx.update(b"\x25\xbe\x46\xe3\xdf\x25\x95\xaa\xaa\x9b\xe9\xd2\x10\x17\xaa\x2d\x64\x30\x67\x20\x48\x8c\x79\xb4\x71\x9c\xe3\xea\xaf\x86\x7d\x4f\x27\x63\x52\xd1\xe3\xab\x46\xd1\x8e\xed\xfc\x79\x78\xc4\xf6\x50\xe1\x06\xfe\x0b\x73\x3e\xc5\xa7\x10\x5d\xcb\xb4\xce\xac\x7e\xf4\x40\x05\xa6\x24\xf8\x43\x00\x1a\x49\xf7\x29\x10\x52\x5d\xbe\x40\x37\x8e");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x27\x6a\x0d\xbb\x54\x2f\xb6\x9e\xd4\x09\xa9\x19\x94\x62\x5f\x84\x69\xb0\x51\xbe\x25\xb5\x96\x8b\x3a\x6f\xb0\xe3"[..]);

        ctx.update(b"\x2b\xf6\x9b\x4b\x50\x31\x2f\x07\xfc\xd6\x8d\x46\x6e\x32\x9c\xea\x51\x5a\xf2\xbc\x9b\x78\xdf\x5e\xb6\x64\xae\x82\xde\x74\x27\x5a\xca\x94\x59\x4f\x8e\x99\x96\xf6\xc0\xd2\xbe\x40\xd9\xe0\xc1\xf1\xdf\xc8\x5f\x53\x41\x47\x32\x2b\x03\x66\xef\xec\x1a\x4d\xe9\x2b\x1a\x5c\xd6\x53\x0e\x78\xd6\x9a\xa7\xa3\xb1\x33\x9a\xd0\x60\xc5\x43\xf5\xd0");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x45\xa6\x61\x10\xf9\x3a\xd7\x87\x39\x23\xb9\x73\x36\xc2\x3e\x87\x64\x2e\x79\xc7\x89\x7d\x90\x11\xdd\xec\xf0\x3a"[..]);

        ctx.update(b"\xa8\x1a\x32\x13\x9d\x69\xfb\xd6\x40\x00\xf8\xd4\xa6\xee\x10\x29\x0c\xcc\x83\xe8\xc9\xe7\x7b\x60\x81\xa2\xe5\x9e\x9c\x15\x7e\x33\xf6\xfa\x31\x37\xd8\xca\xa0\x8d\x28\x23\x01\x69\xdc\x6d\x2b\x10\xb4\xbd\xf0\xdf\x29\xf9\xae\x8f\xa1\x3e\x21\xb7\xe4\x7e\x7b\x92\xd3\x6e\xb7\x03\x58\x9f\x2a\xd6\x99\xea\x5b\x49\x71\x12\x14\xcb\x20\xa5\xb9\x03");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x41\x43\xfa\x44\x8d\xe5\xeb\xc1\x4a\xa2\x04\xd9\x34\xee\x39\x71\x05\x1c\x4e\x30\x74\xd4\xca\xaf\x0c\xc6\x2a\xf1"[..]);

        ctx.update(b"\x47\x1e\x43\x4b\x7e\xf3\x9a\x05\x51\xb2\xed\x92\x50\x00\x63\x53\xec\xd9\x15\xc8\x2b\x77\x64\x9e\xa1\x76\x8f\xe9\x9b\x35\x85\xae\x9d\x7f\xf7\xa4\x0f\x6b\xb2\x3b\x33\x80\x2c\xde\x5a\x1d\x5c\x75\x58\x18\xcf\x2f\x35\xa6\xf9\x7e\x25\x7b\xc9\x63\xba\x32\x80\xeb\x59\xee\xc2\xfa\xeb\x42\xed\x47\xd4\x4e\xab\xa8\x29\x2b\xc3\x90\x1d\x8a\x35\xd5\xbd");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xa1\x2a\x51\xa5\x9f\x11\x6f\x87\xf7\x57\xf2\xfd\xe3\x0e\x11\x98\x0d\x7d\x96\xeb\xc9\x59\x77\x4a\x73\x4e\xde\x74"[..]);

        ctx.update(b"\x3e\x34\x6c\x6a\xaf\x8f\x97\x76\x20\xc3\xac\x68\xfb\x60\x8a\xef\x43\x2d\x6c\xab\x29\x6d\x1a\x93\xda\x62\x61\x94\x9c\x10\xc0\x7a\x19\x43\x64\x14\xfa\xd3\x57\x93\xb8\x27\xf4\x34\xaf\xf8\x9e\x1f\x75\x57\xb1\x8e\x69\xf2\xb9\x31\x8f\x93\xe2\xd2\x9b\xe9\xb0\x35\x4b\x8b\xf0\x42\xfa\x3a\xef\x65\x73\x66\xe9\x36\x03\xf4\xc4\x55\xd2\x3f\xfe\x98\x72\x5d");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x30\x74\x67\x39\x6d\x59\x10\x3a\x89\x7e\x6a\x5b\x16\xdb\xaa\x2a\x0d\xc3\x65\xe2\x23\x95\x46\x5a\x03\x5f\xd2\x93"[..]);

        ctx.update(b"\x1c\xe1\xf0\x1b\x46\xb4\xc9\xe0\x4c\xe4\xb5\xe2\x2f\xbb\xed\x4b\x02\x54\x27\xc5\x58\x48\xb0\x3d\xf9\x8b\xbb\xe7\xd9\x90\xf9\x83\x4a\xb4\x69\xde\x70\xd9\x72\x9b\xf5\x51\x2f\x33\xff\xff\x0a\x3b\x5f\xb8\x8c\xe6\x0c\xef\x73\xca\xcf\x92\x05\xe7\x3f\x2f\xa5\xe5\xf0\x5b\xa4\x9d\xe8\x54\x59\xdb\x52\xe5\x74\x62\x77\x8a\x8f\x31\xe9\xd5\x06\x05\x01\x10\x09");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x5b\xa9\x4e\x26\x39\x44\x85\xeb\x0a\x9e\x3f\x3d\x3d\x9a\x60\x83\xb3\x5d\xae\xd5\xf0\xc8\x89\xbc\x78\x92\x4c\x35"[..]);

        ctx.update(b"\xd6\x24\xf6\x3f\x53\xbe\x3d\x6c\x0a\x20\xc1\x22\x63\xe4\x43\x6e\x46\x24\x07\xd7\x2e\x3d\x85\x42\x80\xd3\x61\x13\x1c\x16\x9d\x95\x31\x43\x0e\x9c\x55\x6d\x0c\x69\xd5\x7d\x6e\xcd\x49\x78\xb6\x92\x6c\x00\x88\x16\x2b\x1e\x61\x39\xfc\x3e\x85\x79\x71\x7b\x39\x5c\x0d\x1d\x33\x0f\x56\x60\x4f\x4c\x2b\x78\x83\x80\x58\xd1\x15\x2e\x68\x9b\x0b\x8e\xbb\x86\xe4\x7b");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x37\xcd\xd5\x23\xd9\x22\x36\x2a\xd4\x63\x21\x6f\x34\x0f\x25\x4e\x76\xaa\xaa\x0b\x45\xf6\xdf\xfc\xa3\x0d\x55\xd0"[..]);

        ctx.update(b"\x7f\x60\x11\x32\xa3\x83\x3a\x3e\x2a\x33\xa0\xf1\xa0\x58\xd6\xa2\xdb\x06\xf1\x1e\x39\xaf\xec\x88\x29\x97\x4b\x64\xed\x89\xff\xee\x9e\xc9\x8a\xb0\x70\x49\x63\x53\x37\x1f\x9c\xb6\x2a\x37\xc2\x3d\xe7\x45\x05\x6c\xb8\xfe\x98\xb4\x15\x88\x5b\x8c\x6a\x2f\xb8\xc4\x1a\x59\xca\x16\xc3\xbc\xc5\xff\xc4\xce\x92\xcd\xfc\x7d\xb9\xf8\xd5\x21\x84\xb5\x81\xaf\x62\xc9\x84");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x10\x87\x1c\x0e\x56\x2e\x11\xaa\x59\x40\x6e\xb4\xbc\xc8\x32\xef\xe4\x94\x22\x60\xef\x08\xcf\xcf\x99\xe7\xdb\x58"[..]);

        ctx.update(b"\x12\xbf\x3a\x33\x37\xd3\xb6\xd5\x53\xbb\x8c\xea\x92\xc3\x69\xb9\x30\x04\x8e\xd2\x90\x2c\x81\xf0\x88\x82\x03\x24\xff\x24\x43\x30\xbb\x93\x51\x31\x74\x72\xd5\x82\x2d\x8b\x01\xdc\xc4\x13\xc4\xc2\xc7\x2f\x41\x1c\x89\xb7\x7f\x22\x75\x39\x62\x7b\xa3\x41\xda\xa6\xa9\xd7\x5a\x9a\xb4\x4a\x8c\xa9\xd7\x2c\x43\xaa\xa2\x9e\xb1\x57\xf7\xce\xda\xf4\x19\xbd\x48\x32\x06\x46");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x6f\x76\x38\x86\xf2\x73\xaa\x5f\x94\x3e\xce\xcf\x7b\x76\x5d\x83\xed\x1a\x78\xb4\x3a\xbe\xf2\x59\x07\x22\x6b\x52"[..]);

        ctx.update(b"\x42\x4f\x1d\x07\xb7\x08\xe6\x0c\xc4\xd0\x03\x96\xcb\x0e\x90\x59\x7f\x39\x77\xae\xa7\x90\x4b\xe6\x1f\x0b\x74\xde\x1f\x91\x8d\xd3\xbf\x6b\x0a\x78\xbb\xf9\x1e\x46\x74\xa3\xdc\x14\x38\x13\x9c\xe7\xc3\xa1\xa0\x0b\x06\x0f\xf7\x63\x27\x77\xa2\xfd\xc8\x81\xf6\xe9\x8c\xd9\x41\x4c\xd7\xe1\xb0\xb0\xd6\xa3\x70\x9e\xc0\x09\x14\x6f\x5b\x50\x37\x5f\xc1\xba\x62\xe3\x64\x83\x30");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x77\x42\xc1\x51\x1f\xc5\xa0\x29\x76\x34\xfb\x3d\x92\x8a\x05\x86\x26\x51\x8b\x7f\xb8\x8d\xb4\x28\xe2\xa2\x82\xae"[..]);

        ctx.update(b"\x87\xe1\xb9\xed\xc7\xc6\x57\x69\x98\xca\x27\xa4\x1e\xcf\x90\x41\x03\x27\xad\xb9\x6d\xb5\xe8\xa1\x2d\x4e\xca\x2c\x3a\x65\x11\xf0\x5d\xe8\x03\xed\x2d\x65\x03\xcb\x3e\x71\x62\x63\x96\x4a\x77\x81\x60\xbb\xf2\xe8\xe5\x40\x6d\x97\xb7\x6b\x44\x4b\x2f\x86\xa7\x45\x1a\x7a\xed\x4d\xb0\xf0\x2b\xf7\xc6\xb5\x06\x25\x44\x63\xc5\xcd\xd8\x3e\xab\xfd\xec\x5c\x83\x1c\xd6\x09\x7c\x72");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x43\x31\xfb\xe4\x8d\x8a\x7f\xd2\xe6\x9a\xb0\x27\x95\x09\x13\x1d\x44\xeb\x13\x24\x27\x31\x17\x24\xe2\xdc\x19\xcc"[..]);

        ctx.update(b"\x8f\xd0\x70\x56\x84\xca\x32\x3c\x8a\xce\x2e\x77\x91\xd4\x07\x69\x7d\x05\xd5\x22\x35\x7d\xd2\x3b\xf6\x4f\x55\x01\xee\xc9\x05\xd7\x06\xfa\x46\xd5\xff\x72\xf8\xb5\xc0\x49\x61\xf0\xee\x63\x67\x24\xbe\xf1\x03\xb9\x07\x3b\xf2\xc4\x9c\x9e\xae\x2b\xd8\xd0\x9c\x0b\x7a\x6c\x87\x56\xb3\xaa\x4d\xc1\x7e\x6d\xd8\xdd\x30\x84\x50\x83\x50\x29\x8f\x76\xa8\xce\xd6\x44\x9c\x1a\xbe\x6d\xbb");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x43\xee\x3b\x23\x4e\xa9\x12\x28\xc0\x53\x58\xfa\x56\xbb\x50\x72\x03\x1f\xa4\x13\xbe\x37\x14\x13\xa2\x57\xa0\xda"[..]);

        ctx.update(b"\xbd\xcc\xba\x55\x40\x77\x61\xa6\xb9\x0a\xe8\x8e\xce\xa8\xb9\x5c\xe1\x3a\xe6\x6a\x34\x06\x6d\x50\x51\x84\xbb\x58\xa9\x6c\x6e\x32\x7b\x14\x5f\xe9\xf7\xf9\xe0\x16\xa4\x9b\x42\x54\xff\x8f\x9b\xf3\x65\xcf\xc5\xf1\x32\x54\xaa\x25\x2b\xe7\xbf\xbe\x19\x34\xcf\xec\xb7\x20\xfa\xdd\xce\x5c\xfd\xeb\x9f\x3a\xb4\x14\x81\xc0\xdd\xaa\xcb\xfc\x1e\x4b\xf4\x96\x4a\x14\xf9\xac\xe0\x12\xff\xd9");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x21\xfa\x86\x6d\x86\x95\x00\x80\x99\xe4\x00\x21\x0d\x04\xa2\x01\x37\x01\x88\x7d\x7a\xfa\x98\xe8\x32\x4e\x96\xde"[..]);

        ctx.update(b"\xa8\xaa\x23\x6c\xde\x55\x7d\x8d\x36\x5e\x6e\xd3\x56\x63\x6e\xe9\x11\x7f\x7a\xe9\xe6\xbf\xb7\x72\xc9\xda\x59\xfb\x27\x1c\xc3\x67\x24\xa4\xef\xd9\x27\xcf\x11\xc9\x98\xf4\x77\x58\xe8\x52\xb4\xe0\xd2\x46\xfd\xd0\xe1\xc2\xd3\xb7\x33\xfa\xe3\x9e\x7c\x60\x2a\x9b\x4d\xd3\x5c\xa3\x97\x13\x87\xa0\x6b\x79\xf3\xc3\x03\xe5\x4e\x1c\x06\xf7\x05\x68\x08\x82\xf5\x2a\x50\x3d\x43\xfd\xdf\xbc\x88");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x10\x2f\xb3\x63\x2c\x67\x52\x24\xb3\x0f\xfa\xa5\xa3\x22\xae\xf3\x35\x4e\x1f\xf0\x18\x3c\xdb\xfd\x06\xe5\xa1\x7f"[..]);

        ctx.update(b"\x44\xc6\xc7\x5e\x37\x7f\x21\xfc\x9c\xd7\xc1\x64\xca\x5c\x4c\xb8\x2c\x55\x38\xa5\x8d\xfb\x32\x39\x92\xe6\xbc\xf5\x88\xc6\x1b\x24\x60\x53\x70\x6b\xf8\x87\x25\xa0\x9d\x0a\x8a\xdf\xcd\xee\xc0\xdb\x41\x9c\xd7\x73\x2b\x0e\x33\x86\xbc\x3f\x34\x07\xe9\xe0\x16\x54\x6f\x4d\x15\xc3\x14\xbf\xd5\x7e\x30\xc3\x02\x92\x6d\xeb\x33\x42\xcb\xc3\x15\xa1\xe7\x06\xc5\x60\x7c\x12\x7d\xe4\x2a\x9a\x73\x9b");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xb9\xb6\x29\x86\xee\xbd\xb3\x5c\x88\xb1\x2e\x02\x57\x53\x7a\x05\x39\x4e\xf5\xa1\x6f\xad\x01\xc2\xfe\xc5\x7d\x6f"[..]);

        ctx.update(b"\xd3\xc4\xa0\xb1\x7f\x46\xcc\x56\x53\xbb\xd8\x30\x0d\xfb\x0d\xf6\xd0\xaf\x3f\xb7\xc7\x63\x9a\x83\x0b\xdc\x9f\x68\xc7\xab\x69\xa0\x9a\x02\x35\x08\x3c\xc7\x21\xd4\x65\xe7\x34\x81\xea\xed\x9f\x0c\x29\xd3\x42\x62\xa8\x17\x72\x17\xf9\x12\xfa\x5a\xe1\x86\x5f\x29\xb9\x76\x69\xc1\x13\x3a\xde\x59\xa1\xd2\x91\x0f\x53\x7b\x9b\x23\xc0\x80\xe9\x63\xc7\x53\xf7\xe6\x76\xdc\xda\xfd\x2a\x11\x84\x67\xf8");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x49\xe4\x55\x3a\x6c\x26\xe7\xa8\x2c\xf7\x0b\x15\x52\x80\x37\xcd\xd6\x58\xae\x33\x65\x3b\xf4\xce\x8e\x86\x79\x9b"[..]);

        ctx.update(b"\xf6\xe3\xdf\x95\xa4\x52\x6a\xaf\x67\x1b\x1a\xb9\x34\x2d\xb3\xba\x0a\x60\x43\xda\x82\xb3\x55\xc4\xa8\xd7\x66\x4e\x09\xf2\x9d\x5c\x0d\x9d\xb6\x01\x87\x2e\x40\xe8\x8b\x3d\x08\xf3\x80\x61\xd5\x21\x75\x3b\x55\xb5\xb9\xb6\x58\x95\xa5\x8c\x3c\xf0\xbd\x0d\xa8\x23\xf8\x1c\x7e\xde\x5b\x7a\x0a\x82\xcc\x47\x80\xce\x26\xc5\xa5\xbb\x89\x60\xcc\xa4\xf4\xeb\x04\xaa\x15\x2d\x90\x57\x5f\x9c\x97\x82\x5e\x68");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x48\x54\xcb\x2a\x3a\x9c\x9e\x3e\xc9\x1b\x4e\x7d\x79\xa7\x53\xf6\xe2\x24\x0f\x4b\x6e\x20\x50\x59\x0d\x6f\x16\x47"[..]);

        ctx.update(b"\xc4\x72\x24\x4b\xd3\x32\x5a\x84\xd3\x72\x20\xe7\xca\x6e\xcd\x40\xe3\xf2\x0e\x52\x67\x7a\x81\x1c\xe8\x98\x7f\x5b\xc0\x96\x39\x50\xba\x5c\x06\xc3\x0e\x95\xfb\x48\x93\x0e\xbd\xc1\x30\x5e\x93\x24\x93\xcc\xbf\x7f\x26\x6d\x0f\x39\x3f\x8d\xaf\x9a\x98\x50\x41\x8d\xfe\x99\x2f\xa5\x4d\x2b\x40\x5f\x9b\x0f\x6a\xa7\xb4\x35\xe8\x9d\xc7\x5e\x8c\x98\x5e\x38\x55\x6b\x93\xcb\x44\xe8\x81\x15\x6c\xa3\xfa\xf1\x49");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xd4\x59\x40\x09\x05\xf4\xc1\xee\xff\x90\xde\xcd\x9c\xc9\x94\xb3\x94\xb2\x14\x30\x37\x9d\x63\x28\xc0\x3a\xf4\x36"[..]);

        ctx.update(b"\x8f\xf0\x52\x08\x05\xa3\x36\xaa\x76\x8a\x1a\x75\xf2\xbe\xc3\xea\x87\x6a\xc2\x24\x6f\x2d\x49\x60\x87\xbf\x3e\xbc\x03\xd8\x8e\x3f\x60\x4d\xc4\x1a\x9b\x7c\xda\x2d\xa1\xeb\x4e\xcd\x06\x2c\x79\xc5\xf9\x04\x1d\x9a\x78\xcc\x3d\x14\xc0\x16\x31\xf9\x08\x94\xf9\x1c\x6e\x36\xd2\x96\x32\x93\x07\x54\x22\x8a\x0a\xd2\x55\x29\xc3\x25\x1c\xcd\x28\x3f\x36\x8b\x0e\x6f\x43\xcf\x3e\x27\x7b\x43\xcc\x02\xa0\xf3\xa4\x05");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x61\x6f\xa8\x49\x5f\x55\xe7\x7a\x30\x8c\x22\x3a\xfe\x47\x0b\x42\xc0\xe1\xa0\x14\x57\x4a\xa3\x6d\x01\xf4\x84\xbb"[..]);

        ctx.update(b"\x6a\x4c\xef\x9b\x8d\xca\x0a\xd7\x76\x75\x15\xc8\xa6\x1a\xc3\x2c\x73\x0b\x28\x94\xb6\x0a\x10\x6b\xc5\x69\x95\x34\xa1\x70\xf1\x13\x70\xff\x90\x8f\x2d\x74\x6a\x96\x9e\xcf\x8f\xc1\x86\x15\x03\xda\x1d\xf2\xcd\x9c\xbd\xa1\x4b\xb9\xde\x11\xc8\x22\x44\x40\x50\x82\x77\xeb\xc3\x0b\x74\x6a\x29\x76\xa2\xd6\x45\x45\x07\x6c\x6d\x4c\xbc\x36\x99\x6f\x9e\xe8\xb9\xad\x47\x08\x4c\x5e\x3d\x51\x2e\xb8\x68\x5a\xb1\x98\xca");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xd2\x04\x90\xeb\x96\x33\xa3\x21\xa6\x9e\xb2\x7d\x42\xdd\x61\x9a\xce\xee\x53\x3e\x85\x00\x75\xfd\x16\x58\x9b\xef"[..]);

        ctx.update(b"\x9a\x4f\xd4\x4a\x08\xd9\xf5\xe7\x73\x16\x03\xac\x92\xb1\x8e\x3d\x88\x0f\xf2\xd5\xb9\xd7\xce\xe6\xdb\xb2\xb7\xff\x50\x4f\x2d\xf3\xb3\xab\xcd\x38\xb1\x8f\xc9\x8d\x1a\x5a\x96\xdf\x37\x00\xe4\x6e\x5f\x53\xd3\x2c\xc5\x81\x64\x65\x94\xde\x2d\xa5\x3e\xe4\x52\xd7\x10\x78\xb0\xaa\x0b\x6f\x5b\x39\xef\x51\x4b\xea\x98\x5d\x52\x96\x8d\xcd\x51\xeb\xd7\x5b\xac\x0f\x0f\x03\xfb\x86\xbb\x0b\x03\x56\xdb\x41\xe4\x69\xbe\x84");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xbf\x3a\x90\xe1\xe0\xe8\xb1\xa2\x0b\x08\x33\x1d\xa3\xc1\x99\x6c\x83\x95\xa1\xe3\xb8\x35\x79\x9f\x87\x9c\xc7\xdf"[..]);

        ctx.update(b"\xaf\xce\x17\x3b\x05\x7b\x48\x40\x6e\x16\x35\x84\xc8\xd8\xc1\xa9\x3b\x15\x80\x46\x12\xc2\x27\x8d\xca\x34\x41\x2b\x69\x2a\x24\x70\x12\x0e\x24\xb0\xd5\x78\x06\xe9\xcb\xf4\xb8\xa5\x63\x05\x3f\x22\xf7\x6a\xf0\xb1\xa3\x9f\xe3\x5c\x56\xa7\xa5\xcb\x29\x09\x2d\x69\xf0\x19\x11\xc2\x0f\x78\xfa\x2f\x90\xde\x56\xb6\x72\x17\x4c\x2c\x43\x4c\xf5\x9c\xf1\x84\xc9\xed\xdd\x6a\x57\xbe\x31\x51\x3a\x95\x74\x75\x48\x39\x5b\x84\x0c");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x37\x3d\x08\x13\x91\x39\xaf\x96\xf2\x28\x10\xc8\xe8\x44\x3a\x78\x1c\x88\x34\x4b\x73\xb9\xb7\x17\xa9\x7f\x4a\x04"[..]);

        ctx.update(b"\x54\x21\xcb\xce\x30\x7d\x2c\x89\xa2\x7a\x7b\x0a\x1d\x51\x5e\x69\x27\x00\x2b\xb6\x5d\x3f\x09\x90\xb8\xb4\x52\xf3\x10\x6a\x3c\x52\xde\x80\xb9\xe6\x46\x1e\x9b\x45\x55\xad\xfc\x36\x87\x48\xdd\xc4\x8c\x6d\x0d\x91\x4b\xf9\x4d\x6d\x28\x53\xb1\x27\x82\xe7\x49\xa3\x80\xd6\x01\x2d\x0e\xeb\xc6\x0d\x22\xd2\xea\x5c\x44\x84\xca\x16\x4d\x06\xae\x70\x6f\x53\xf0\x6b\x75\x97\x61\xa9\x4c\x9d\xe8\x7f\xc3\x8b\x6c\x4d\x8f\xcd\xf4\x0a");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x5b\xc1\xdc\xf5\x93\x42\x49\x91\x95\xa6\x1a\x06\x61\xaf\x7a\x8e\x96\x09\x76\x60\xa9\xf7\xe5\x76\x57\x2f\x3d\x0b"[..]);

        ctx.update(b"\x5a\xdd\x90\x71\xd1\xd5\xd1\x79\x9c\x0f\x92\x23\x10\x9b\x76\x1b\x7b\x9e\xe1\xc3\x0a\xa2\x00\x25\xea\x30\x0a\x2b\xae\xa1\xc5\x12\xf1\x31\x4b\x1c\x3c\x55\x0e\x85\x2a\x4f\x1a\xb9\xc9\xff\x9a\xc4\x44\xfc\x9e\x0a\xbf\xe3\x1b\x38\x83\xef\x33\x7a\x5b\xca\xb7\xef\x3d\xa3\xeb\x5a\x2d\x81\x89\x1a\x5f\xd7\x08\x5c\x9c\x58\x1d\x8f\xe6\x0c\x83\x18\x5f\xa4\xf8\x47\x67\x0b\x88\x8a\x05\x3d\x0d\xe1\x88\x5c\xf7\x92\x6e\x01\x49\x1e\x84");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x0c\x37\x04\x10\xdc\x8e\x22\xee\x61\xc4\x9d\x7a\x3b\xd3\xd7\xe8\x0c\xc4\xf3\xf5\x1a\xcb\x26\xae\x4e\xc5\xa3\xa1"[..]);

        ctx.update(b"\xe9\x65\xba\x8d\x12\x36\xa1\x0b\x36\x2c\xd7\x5c\xac\xd0\x17\x5c\xad\x0a\xf6\x5e\x13\xd9\xc5\x7a\x60\x47\xb7\x29\x02\x9d\x47\x0f\xd6\xd3\xd4\x17\xea\x4a\x8e\xc5\x87\xbe\x96\xc8\x6f\x76\x61\xcf\x63\xbd\x5e\x9e\xe8\x96\xd8\x61\x4b\xac\x1d\x40\xe3\x88\xe0\x33\x96\x06\xdc\x18\x9e\x0a\xba\x03\x8d\x6f\x35\xd5\x8f\x40\x7c\x10\xf0\x2b\xd7\xe9\xde\x32\x8e\x9f\x6f\xa4\x07\x0c\xe6\x4c\x2e\x7a\x32\xe8\x19\xb7\xb7\xdd\xed\x6c\x51\x28");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x17\x5c\x4a\x7f\xa4\x91\xe7\xdb\xe3\xa2\xc8\x60\x4b\x07\xba\xb0\xe2\xa2\xe3\x4b\x27\x0e\x16\xb4\x39\x2e\x97\x16"[..]);

        ctx.update(b"\x31\x41\x58\x33\x6f\xbf\x78\x45\x25\x1e\x1a\x62\x06\x13\xad\x8c\x1a\x59\x96\xe5\x12\xae\x25\xcd\xd9\x1e\xc8\x4b\x6b\x83\x7f\xae\xea\x4d\x90\x74\xef\x7f\x7c\x6a\x1e\xfa\x31\x15\xfd\xfe\x81\x51\x06\xee\xd3\x38\xfa\x53\x19\x45\x79\x24\x02\xb9\xd0\x45\x70\x3e\x12\x4d\x2c\xcb\x6a\x8c\xd9\xc7\xaf\x4d\x26\x71\x39\x58\x48\x7f\xf3\xbb\xa8\x05\xb2\xca\xd8\x99\xc5\x65\xcc\xb5\xce\x79\x47\x5d\xd0\xc9\x93\x01\x88\xef\x75\xa4\x01\x9d\x60");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xd6\xcd\x19\x01\xbb\xea\xf4\xb6\x9a\x7c\x21\x1c\x50\x54\x6f\xce\x35\x38\xa8\x38\x84\x60\xec\x29\xff\x81\x66\xdf"[..]);

        ctx.update(b"\x2c\x2d\x86\x19\x7b\x39\xc1\x94\xc6\x5a\x93\xbd\xad\xc7\x2a\x9e\x59\x0f\xa7\x5f\xa8\xae\x6b\x75\x8a\xdf\x91\x16\xab\xb2\xbb\xf0\x52\x5e\x12\x1a\x89\xd8\x88\x4b\x7e\xa5\x61\x4a\x29\xec\xf1\xa5\x2f\xcf\xa7\xa6\x06\xfb\x4a\x83\x5a\x7e\x83\xd5\xb0\xda\x9c\x4b\x4a\x57\x87\xa0\x4a\x9d\x22\xca\x1a\x81\xb9\x75\x0a\x20\xcc\xaf\x3b\x2a\x13\xe5\xbd\x81\xc0\x0e\xf4\x03\x04\x26\x40\xa3\xd4\xde\xdc\xbf\xed\x92\x07\xae\x16\x7b\x62\x98\xa5\x26");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x49\xda\xe2\x1e\x77\x33\xce\x09\xe3\x92\xdd\x98\x0e\xc3\xa0\xc2\xc5\x42\x69\x37\x6e\x23\x1c\x19\xdc\xf5\xa7\x74"[..]);

        ctx.update(b"\xbc\x10\x6b\x07\x1b\x53\xc4\xd8\x55\xe1\xa3\xdd\xa4\x1d\xb1\xa7\xa4\x46\x05\x86\xb4\x48\xa3\xfd\x9f\xd8\x42\x74\x7c\xbf\x53\x2d\xd2\x76\x3e\x79\x00\xf4\xef\x2c\xdc\xc4\x91\x4e\x9b\x5a\x99\x13\xf1\xa7\x07\x0d\x78\xeb\xe8\xa1\x56\xcb\x20\xb8\x1b\xa3\x66\xb6\xe8\x1d\x90\xb0\xd6\x03\x11\xda\x95\x51\xc0\x3e\x54\x8f\xdb\xb4\xe1\x5d\x4b\x0e\x12\x50\x29\x93\x63\xfc\x82\xaf\xe0\x52\x57\xe8\x46\x2d\xe8\x14\x65\x27\x33\xc9\xc8\xb1\xb7\x16\x98");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x3b\xaf\xf8\xbc\x1f\x43\x51\xfc\x62\xca\x40\x90\x4a\x00\x21\x98\xb5\xef\x69\x24\xaf\xba\x1d\x1f\xc6\x1d\x8a\xe1"[..]);

        ctx.update(b"\xf6\x27\x84\x8c\x4a\xfe\xcd\x8d\xf3\x3d\xa1\x9c\x60\x6a\xff\x11\x9a\x37\x62\xc6\x1c\x05\x39\xc3\xe6\xa8\x96\x1b\xcd\x6a\x2a\x11\x15\x52\x71\x7f\x41\xa6\x2a\x79\x2a\xae\x0a\x2d\x4f\xbb\xb1\x0d\x6c\x68\xdd\x31\x4e\x20\xde\xf6\xe5\xe0\x4a\xec\x9b\x3e\xf1\xef\xe4\xab\x13\x93\x16\xf5\xd1\xc6\x7c\x10\xe2\xb0\xa5\x15\x87\xde\x02\x8b\xc1\xa0\xb1\xe6\xc1\xaf\x93\x4d\xa0\x13\xa4\xa0\x50\x44\x8c\xba\xd3\x90\x71\xf3\xaa\xbc\x10\xe9\x91\xee\x0c\x81");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x2e\x0b\x54\x55\xcc\xe6\x92\x62\x06\x6f\xdb\xf9\x39\x0a\xd0\xdc\x56\x13\x8b\x22\x1b\x75\xd5\x12\x98\xbb\x5b\x2e"[..]);

        ctx.update(b"\x15\xcb\x77\x7e\xf3\xe4\x51\xb9\x28\xdb\xf2\x88\xe4\x6a\x36\x27\x04\x4f\xf5\xde\x42\xad\xd8\x84\xa9\xaf\x6b\x42\x4d\x6e\x73\x99\x38\x15\x81\xa6\xa7\x43\xc7\xa5\x77\xb0\x2b\xb5\xda\x14\x9a\xda\x4e\x44\x9f\x48\xd0\x9e\x34\xdf\x4c\xa8\xd8\xf2\x59\xf4\xe1\x4c\x23\x47\x14\x75\xa8\xf9\x73\x31\x28\x9f\x56\x4a\xd6\xe8\xbd\x8f\xd4\xc5\xe5\x1d\x5e\xcd\x19\xdd\x46\xdf\xcb\x4e\xa0\x09\xe3\x85\xbe\xa8\x57\x72\x5f\xd1\xfc\x64\x23\xf0\x9c\xcf\x42\xaf\x48");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x9f\xa1\x25\x61\xf1\xdf\x9a\x2d\x79\x32\x92\xe0\xf4\xdf\x53\x27\xaf\x52\x93\x36\xb5\xb2\x11\x89\x52\xf5\xc2\x4e"[..]);

        ctx.update(b"\xd2\x4d\xf7\x5a\x00\xcf\x92\x67\x7b\xb4\x1a\x62\x0f\xae\x51\x97\x23\x93\x7e\xbf\xe1\xf7\xb4\x30\x97\x00\x56\x50\x5d\x76\xdb\x4f\xf9\x1a\xcf\x16\xff\x39\x1a\x7a\x3d\x80\x85\xb6\x55\x12\x7a\x18\xac\xd8\x0b\xfa\x83\x18\x37\xf4\x64\x4a\x68\x50\xc0\x27\x3f\xbe\xd6\x02\x94\x49\xd6\x5b\xb9\x8a\x47\xb2\xff\x1c\xa6\x99\x7c\x50\x50\x0d\x0b\x21\xa2\x06\x93\x6a\x5e\x4d\x8d\x56\x50\x8e\xc0\x18\x32\xae\x4f\xdd\xce\x5e\xf6\xff\x62\xf1\x91\x7c\x48\x6a\xde\xa6");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x5c\xae\x12\xea\x96\x52\x26\x9e\xa2\xaa\xfc\x65\x6c\xb8\x34\x24\x74\x6e\xa1\xd5\xd4\x91\xf9\xa1\x59\x59\x4b\x2a"[..]);

        ctx.update(b"\x73\x09\x63\xb1\xa4\x62\xd2\x34\xe8\x70\x5d\xb9\x27\x8c\xc5\x8d\xcb\x81\x7b\xec\x6c\x9e\x62\xbc\xad\x0a\xd5\x7c\x61\x58\xec\x77\xf3\x5b\xfb\x3d\x04\x3c\x5c\x53\x55\xc9\x6f\x2a\xe8\x10\xde\x4e\x62\x2d\x26\xe0\xb4\x60\x53\x46\xfa\x63\x0a\x21\xa3\xfa\xcd\x7f\xd3\xcb\x7b\x53\x05\x70\x16\x22\xf5\xcb\xc9\xfe\xbd\x99\x2a\xb8\x3e\x4a\xbc\x52\xc1\x11\xb8\xb3\xde\x37\x0d\x9e\x9e\xb4\xa5\xe5\xdc\x00\xea\xda\xf1\x27\x89\x07\x90\x1c\x75\x1f\xee\x4f\x30\x34\x26");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x46\xc7\x04\x6a\xa6\x33\xfb\x5b\xc4\xd6\x95\x2f\x85\x3e\x39\xc5\x3e\x77\x10\x09\x18\x3b\x66\x96\x15\x93\xa3\xcf"[..]);

        ctx.update(b"\xc1\x46\x0b\x2e\x1e\x38\x5c\x0b\x57\x56\xe2\x10\xbf\xfa\xa8\xf1\xcb\x12\xeb\xc3\xc6\x0e\x71\x05\xc5\xfd\x1c\xe0\xde\xf2\xbe\xae\x95\xad\x2f\xba\x97\x77\x9d\x46\xab\x4e\x24\xd3\xe5\x42\x9e\x9c\x1c\x1f\xbd\x64\x9f\x4c\x57\x71\x24\xf3\x00\xd9\x01\x62\xde\xfc\x9f\xb2\xb9\x52\x2b\xa6\x52\x67\x8e\xd5\xf6\x23\xa0\x0c\x52\x4f\x98\xee\x89\x55\xc8\xdd\x84\x47\x87\x1e\x8a\x23\x6e\x37\x10\x56\xc7\x9c\xd2\x7e\x42\x8b\x22\x74\xea\x35\x80\x8a\x58\x1e\x6c\x11\x26\xc6");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xd8\x40\x2e\xe6\x34\xe1\x8c\x96\x7d\x11\x0b\xd7\x05\x3a\x30\x2d\xa6\x50\xcc\xb3\x8f\x25\x9c\xbe\xe5\xa2\x47\x15"[..]);

        ctx.update(b"\x48\xa8\x4e\x3c\x48\x50\xa2\x9e\x43\x3c\x8a\x24\x3b\xf4\xbe\xb0\x5b\x8a\x5b\xff\x38\xcc\x41\x81\x58\xbf\xb9\x68\xf1\x49\x5c\xff\xf2\x2e\x30\x3e\x47\x3e\x13\x9b\x59\xcc\x5d\x22\xd5\x61\xc1\xc8\xaf\xb2\x74\xfd\x66\xd0\x91\x80\x89\x9f\xbd\x77\x7a\x89\x07\xbf\x72\x73\x11\x81\xc5\xf0\x75\x06\x68\xb5\x62\xcc\xb0\x4e\x86\xd9\x5c\x0f\x31\x4f\xd6\x4c\xb7\x41\xb6\xcd\x16\xc5\x88\x7c\xd8\x4a\xcf\x61\x3b\x2c\x4a\xd0\x90\x86\x75\xe1\xe1\x48\x53\x21\xd6\x89\x6b\x87\xd9");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x37\x48\x80\x8c\x1a\x4d\x1d\xc1\xb8\x2e\xba\xb4\xcf\xfc\x9d\x8c\xf5\xc2\x51\xd0\x45\xdc\x8a\xff\xd7\x56\xda\xe3"[..]);

        ctx.update(b"\x41\xb2\xb0\x71\xee\x6a\x8e\x0b\xc6\x2e\x9e\x03\xa8\x70\x5f\x6d\xa2\x18\xe6\x9b\x11\x1c\x23\x4c\x0a\x51\x65\x20\xf9\x81\x20\x4f\x88\xbd\xef\x9b\xc3\xf4\xd5\x83\x79\xc5\xc8\xba\xe5\x92\x70\xd7\x8d\xd6\x2c\x4e\x6b\xb8\x56\x9f\x4b\xe0\xc8\xcd\x74\x89\x9e\x12\x8a\xd7\x3e\xd1\xb3\x6b\x96\xcc\xf9\x36\xa5\x97\xf0\x8a\x3d\x4d\x98\x77\x07\x81\x84\x54\xc7\x77\x3c\x9c\xe1\x94\xe1\xfc\x1d\x14\x0f\xa9\xa7\xfb\x07\xf7\x5a\x4e\x0e\x28\x3d\x82\x10\x81\x4e\x35\x89\xa4\x7f\x1c");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x87\x76\xf0\x80\x70\xe3\x85\x82\xd4\x60\x41\x70\xa9\xe6\x58\xfc\x17\x03\x4c\xbd\xe9\xe0\x3a\x00\x2e\x72\x81\x31"[..]);

        ctx.update(b"\x54\xbf\x37\x59\x44\xc0\x02\xcc\xac\x87\xf5\x5f\xf3\xeb\x58\xc4\x58\x44\xf6\x2d\x91\x7d\xdd\x03\xb2\x6b\x5f\xdb\x51\x5d\x5c\xa6\xe8\xf5\xc4\x27\x7a\x36\x57\xab\x7a\xdd\xaf\xf4\xe8\xf5\x81\x76\xbd\xa6\x20\xc6\x89\x4e\x42\x4f\x5b\xd3\x07\x47\xe1\x37\x37\xe1\xd0\x6c\xcb\xc4\x86\x6a\xb0\xbd\xef\x80\xc7\xb0\xa5\xe7\x30\xd5\x5b\x9d\xc8\x20\x7b\x70\xf0\x01\x0e\x59\x91\x4e\xdd\x0c\x5d\x7e\xd5\xcd\x57\x17\xb0\x85\x95\x69\xd2\x15\x44\x7d\x71\xea\xee\x49\xf9\xd2\xf8\x25\xaf");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x0c\xd6\x90\x73\x6d\xd1\xf3\x90\x69\x4f\x65\xc3\xb8\x9e\x0c\xc2\x26\x9f\x3b\x82\x08\xc8\x06\x92\x9a\x8f\x8c\xb9"[..]);

        ctx.update(b"\xd9\x24\xb1\x6d\xfe\x2f\x61\x2d\x92\x3b\x25\x66\x9b\x45\xd9\x11\x7d\x30\xa3\x53\x1d\x5d\x30\xad\x47\x7c\x41\x73\x4d\xa2\x67\x7f\xf1\x9b\x0a\x26\x85\x81\xb0\x6b\x51\x8d\x7c\x73\x35\x2e\xf5\x39\x90\x58\x0a\x83\x60\x1f\x52\x15\xaa\x84\x98\x24\x1c\x8f\x96\x0d\xf1\x9e\x01\x47\x15\xfc\xc6\x96\x89\xb5\x94\x82\x0a\xa9\x9a\x08\xed\xe0\x92\x8f\x4e\x1d\xfe\x1e\x69\xb6\xb1\xcb\x48\xa2\x92\x77\x9f\xd4\x9d\xb8\xf4\xc7\x2a\x17\x02\x5b\x72\x7c\x60\xdf\x4f\xe6\x5e\x6f\x43\x5f\x30\x80");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x30\x4e\x38\x68\x92\x50\x38\x0e\x32\x25\x05\xb4\xe2\x3f\xff\x35\xb3\x41\xe2\xd9\x4c\xc6\x2c\xfb\x44\xb5\x1d\xb5"[..]);

        ctx.update(b"\x7d\x2d\x12\xb3\x51\xa7\x03\x06\x23\x0e\x7d\xcc\x76\x64\xca\x38\x83\x92\x80\x93\x39\x5b\xa8\xd5\x82\x8f\x1f\x76\xc8\x81\x3d\x82\xb8\xba\xb5\xec\x45\xef\xbc\x24\xf0\x31\x6e\xeb\x47\x64\x6d\x67\xac\xf2\xcb\x72\x58\xc1\x2a\x38\xa0\x62\x21\x01\x76\x8b\xe2\x4b\x32\xc7\x1a\x1c\x93\x6c\x19\x96\x00\xcd\x33\xcc\x14\xb2\x66\xd4\x5a\x05\x3d\x1e\x55\xce\xf3\x03\xc4\x62\xdc\xa1\x23\xd6\x01\xe7\x75\xba\x0a\xec\x41\x04\x62\x85\xdb\x4b\x95\x16\xe4\x36\xfd\xa8\x14\x33\x9d\x3c\x37\x6c\x4d");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x40\xa7\x94\x7a\xad\x6c\x1e\x73\x43\x6d\x78\x67\x20\x38\xfb\x99\xa4\xed\x7c\xb1\x45\x44\x5e\x27\x66\x43\xcf\x38"[..]);

        ctx.update(b"\x5f\xee\x68\x78\x02\xa5\xaa\xc1\x33\x29\x17\xcc\x5d\x0a\x94\x39\xad\x95\xc7\x64\x21\x9b\xce\x45\xe2\x74\x30\x94\xd3\x44\x13\x30\x6a\x7d\xb0\xc1\xbf\xa5\x05\x9c\xe2\x5a\xc9\xe8\x72\x93\xa5\xff\x6b\x28\xa1\x17\x40\x95\xdd\x22\x2a\xfb\x81\xf3\xbf\x67\xf6\xa1\xc6\x57\x57\x30\x4a\x7a\x22\x6e\xed\xb8\x34\x1c\x7f\x72\xf4\x4e\xcf\x0b\xce\x15\xc1\xa7\xe3\x5a\xa1\x32\xfe\x0d\xb7\xa5\xca\x80\x1c\x84\x43\x0d\xcc\xab\x6b\xff\xfb\x4b\x3e\xb0\xad\x30\xe7\xee\x81\x45\x34\xa3\xcd\x59\x48\xb5");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xd6\x65\x7f\x11\x5d\xb7\x59\x44\x47\x44\xa1\x25\xe4\x0a\xde\x36\x1d\x8a\x40\xfd\x88\xc9\x11\x87\xdc\x5a\x50\xb3"[..]);

        ctx.update(b"\x69\x67\x30\xb7\xec\xa2\x70\x66\xd5\x41\xc4\x15\x0a\x22\xe6\xa6\xb0\x62\xc9\x51\xf2\x17\x7d\xf6\xfd\x42\x54\xb1\x4f\xff\x43\xb0\x32\xff\x8a\x41\x83\x38\x97\x2f\xb8\xdb\x28\x4f\x22\x87\xb1\x56\x6e\xbc\xfd\xed\x72\x92\x17\x08\x27\xab\x6b\x73\xf1\x66\x29\xad\x27\x55\xea\xfa\xcf\xf2\x96\xb4\xa9\x8d\x85\xe2\x70\x89\xe4\x0c\x5c\xc8\x5b\x4d\xeb\x64\xec\xa9\xe1\xa0\x1f\xa1\x8d\x1e\x5c\x05\x4f\x52\x98\x0a\x57\x36\x34\x9c\xaf\x3f\x21\xf5\x7c\x90\x49\x6f\x81\x6f\x2a\x96\xe4\xf5\xc4\x82\x35");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x39\x1d\x6c\x2b\x64\x5d\x1e\xaa\x7a\x7d\xd6\x81\x40\xfb\xb7\xc2\x3d\xd4\xac\x03\x47\x88\x7a\x78\x2e\x4c\x34\x42"[..]);

        ctx.update(b"\xc8\x99\x5c\xe0\x9a\x8d\x80\xb7\x90\x78\x99\x15\xb7\x7e\x53\xcb\x2b\xa0\xb3\x97\x43\x27\xf0\xa5\x09\x46\x85\x55\xed\xfb\x84\x3f\xba\x36\x2d\x75\xc8\x3d\xbf\x07\x74\x4f\x8e\xcd\xcd\x87\x56\x60\x90\x04\xcf\x6a\x57\x47\x81\x20\x0d\xdc\xda\x0c\xa5\x3d\x7e\x10\x0b\xf3\xde\xa8\x5b\xfe\x0c\x6f\x52\x77\xce\xe5\x02\x74\xf1\x6f\xc9\x2c\xdf\x9f\x83\x5c\x74\xb6\xff\xe9\x0e\x67\xb0\xc6\xe3\xde\xa5\x18\x98\x01\xbb\x98\x0f\xf6\x12\x39\x80\xe5\xbd\x0c\xcb\xff\xbe\x60\x0c\xbf\x31\x2d\x70\x6f\xc7\xb3");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x3a\x6a\xd1\x6c\x59\x57\xb5\x4e\xe6\x69\x1d\x22\x7b\x2d\x2e\xcf\x93\x48\xb6\x84\x35\xa5\x06\x47\xa2\xcd\x55\xc9"[..]);

        ctx.update(b"\x24\x7f\x87\x54\xb0\x08\xac\xea\x23\x4d\x3b\x5c\xab\x5c\x61\x3a\x69\x5a\xf8\xfe\xbd\xbc\xf3\xe4\xc5\xbb\xfd\xfb\x20\x01\x5d\x7c\xf8\xa9\x1d\x5b\x3f\x66\x6b\x1d\x38\x02\x3f\xe9\x3a\x0b\x3e\xe1\xb1\xc9\xdf\xb3\x65\x72\xcc\x6a\xff\xa1\x8b\x8b\xc9\x05\x5d\x5d\xb5\x56\x66\x88\xa1\x8e\x0f\xe1\xd9\x55\xa9\x85\xdd\xbd\x53\x25\x65\x2f\x1d\xa0\xa3\xa6\x6d\xd7\xc8\x1a\xa2\xbd\xc9\x40\xf8\xad\x63\x70\xab\x01\x5d\xe6\x60\xe0\x0d\x2e\xc0\x57\x1d\xda\x66\xab\x8f\x76\xfb\xee\xff\xbe\x9f\xd1\x28\x0e\x8b");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xfe\x66\x02\x6c\x1c\x24\x10\x92\x7f\xa4\x42\xcc\x12\x2d\x5a\xed\x90\x30\x82\x73\x40\x1d\x42\x9f\xd1\xac\x8f\x61"[..]);

        ctx.update(b"\x93\xdb\xad\xa7\x1b\x89\x0e\x80\xd9\xb8\x8a\x33\x8a\x7d\xf0\x20\xa7\xbf\x9f\xfd\x98\x98\x53\x62\x25\xfd\x3e\x8a\x2d\xa2\x5b\x58\x9a\x18\xff\x98\x94\x38\xb1\x88\xf5\xc5\x3b\x96\xfd\x4d\x01\x6c\x48\x50\x92\x2a\xfc\xa5\x7d\xf2\x47\x18\x66\xaa\x09\x58\x58\xfd\xcd\x11\x5d\xa3\xfe\x55\x4c\xe3\x58\x40\xe3\x56\x9c\xdc\x15\xed\x88\x42\xa5\x35\x06\x02\xf4\xda\xa4\xc5\x18\x6b\x44\x31\x2d\x5c\x14\x0d\xf0\x51\xc1\xa6\x8a\x9a\xe8\x71\xa9\xf9\xce\xf1\xd5\xac\x50\xdb\x02\xa7\x1d\x79\x61\x22\x22\x4c\x7c\x53");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xf7\x31\x54\xe7\x6b\x59\x98\xe0\x79\x2f\xb0\xab\x1d\x72\x1e\xe7\x30\xb6\x4a\x1b\x3a\x54\x30\x61\xfc\x76\xcd\x17"[..]);

        ctx.update(b"\x61\x78\x73\x6e\xa6\xaf\xf4\xb9\x9e\x7d\xa9\x7d\xee\x00\x0a\xdf\x72\xb3\x70\x69\xdf\x8a\xd5\xbb\xa6\x3f\xdb\x7e\x1b\xe6\x10\x9a\xd6\xc8\x71\xa9\x59\x6e\x2d\xab\xf4\xfc\xab\xfb\xe6\xc1\xcd\x82\xc5\x0b\x77\x2a\x20\x40\x4b\x95\x62\x2a\xb7\x10\xf9\x93\x32\xc2\xb8\x12\xc2\x29\x7a\x24\xc2\xc8\x09\x7e\xe2\x19\x3e\xa2\x6f\xc9\x69\xb1\xd5\x24\xf2\x80\x62\xcf\x41\xed\x4c\xb5\xdc\x74\xf6\x43\xcf\xfd\x6f\xd8\x51\xf8\x3b\x77\xa4\xac\x57\xc0\x3d\xcc\x77\x1e\x25\x4c\x92\x41\x9f\x72\xa0\xcf\x0b\x30\xa3\x56\x0a");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x33\xa7\x42\x53\x16\xd2\xf6\xbe\x1b\xdb\xb8\x70\xd9\x53\x13\x2d\x55\xe2\x9f\xdf\x85\x30\x46\x39\x77\x94\x09\x08"[..]);

        ctx.update(b"\x1c\x21\xb5\x84\x23\x12\x11\x1a\x9e\x27\x82\xfe\x97\x91\x7e\x94\x6b\x8d\x65\x98\x7c\x43\xc9\x1b\x27\xed\x0c\xc8\xf6\x3b\x31\x6c\x16\xf6\xec\xb6\x0b\xe5\xfb\x3b\xa2\xed\xb9\xbc\x27\x7b\xf1\xac\xd9\x27\xf0\xed\x9b\x91\xde\xdb\xec\x8f\x4e\x8b\x19\x8b\x13\x54\xb1\x26\x55\x8f\x3a\x21\xe5\x9c\x22\x7e\xce\x26\xf9\xe2\x2d\xa7\xdd\x3e\x42\x2e\x35\xae\xc6\x60\xa2\xa2\x7f\xbc\x32\x3b\x13\x24\xd7\xf8\x1a\x99\x18\x91\x76\x47\x07\x23\xe7\xc1\x73\x8e\xda\x8d\x5e\x1b\x82\xe3\xb1\x63\x2a\xb2\x2f\x6c\xca\xe4\xa8\x14");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xdb\x43\xca\x3d\xd7\xbc\xa3\xd8\x99\x64\x86\x92\xe4\xed\x8f\x03\x11\x96\x28\xdf\x31\x3e\x15\x73\x72\x3c\x58\xe9"[..]);

        ctx.update(b"\x4b\x98\x95\x23\x5c\xb4\x95\x6a\xef\xff\xe8\x15\x41\x52\x52\xe7\xd6\xb2\x19\x21\xbd\x7f\x67\x53\x15\xef\xf0\x71\xd0\xbb\xd4\x29\xb7\x18\xc7\x74\xae\xe9\x6f\x6c\x3a\x33\x0d\x5d\x40\xd1\x60\x1e\x10\x69\xc7\xa2\xa1\x9e\xa5\xca\x1e\x87\x09\x7d\xa2\x60\x8f\xfb\x41\x80\x81\x6e\x47\x8b\x42\xc3\xc4\xe9\xed\xb7\x48\x77\x39\x35\xeb\x7c\xa0\xdf\x90\xde\xc0\xeb\x6b\x96\x01\x30\xc1\x61\x78\x80\xef\xb8\x0b\x39\xae\x03\xd6\x17\x95\x0a\xce\x4c\xe0\xac\xa4\xd3\x6f\xd3\xed\x01\x12\xa7\x7f\x5d\x03\x02\x1e\xb1\xb4\x24\x58");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x9a\x91\x76\xe9\x7a\xec\x99\xab\x07\xf4\x68\xf6\xa2\x26\x87\x67\x10\xd6\xd8\x77\x02\x1d\x27\x06\x1d\x4d\x01\x32"[..]);

        ctx.update(b"\x9e\x12\x78\x70\xbe\x24\x31\xbc\xb4\xf4\xeb\x4e\xfd\x5c\x2a\x6c\x58\x70\xc5\x5e\x7a\x5e\x3b\x75\x03\x99\x4a\x4c\xb1\x36\xbe\x4e\xd3\x96\x88\x78\x01\x45\x0f\x60\x0b\x22\xcb\x77\x2f\xc0\x0f\x8b\x8f\x0d\x26\x90\xe2\x31\xa2\x9f\x69\xb9\xf1\x3f\x24\xf5\x31\xe4\x47\x9e\x45\xb5\xe8\xbc\x29\x92\xfa\xc7\x82\x56\x7e\x0d\x7a\x59\xf8\x53\xca\x3a\x20\xbf\x18\xdb\xdb\xf6\x84\xac\x69\x81\x7e\x2d\xe0\x75\xda\xae\xd9\x53\x26\x59\x69\x2d\x3b\x73\x53\x0a\x12\xdf\x7b\x8c\xd9\xe4\x9e\xd0\x46\x30\x41\x96\x2c\x1c\xe7\xa2\x4c\x31");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x7e\x2c\xf6\x22\x66\x23\x53\x57\x84\xc5\x9c\xd6\xa7\xb2\x7d\xac\x60\xee\x23\xfd\xce\x8a\x80\x4d\xbd\x6d\xfe\xdd"[..]);
    }
}
