# Copyright 2007 One Laptop Per Child
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

import logging
from gettext import gettext as _

import dbus
import gtk
import hippo
import telepathy
import telepathy.client

from sugar.activity.activity import Activity, ActivityToolbox
from sugar.presence import presenceservice
from sugar.presence.sugartubeconn import SugarTubeConnection
import sugar.logger

import gridwidget
from game import ConnectGame, SERVICE
from buddiespanel import BuddiesPanel
from infopanel import InfoPanel


logger = logging.getLogger('connect-activity')


class ConnectActivity(Activity):
    def __init__(self, handle):
        Activity.__init__(self, handle)

        logger.debug('Starting Connect activity...')

        self.grid = gridwidget.GridWidget()

        self.buddies_panel = BuddiesPanel()

        self.info_panel = InfoPanel()

        vbox = hippo.CanvasBox(spacing=4,
            orientation=hippo.ORIENTATION_VERTICAL)

        hbox = hippo.CanvasBox(spacing=4,
            orientation=hippo.ORIENTATION_HORIZONTAL)

        hbox.append(self.buddies_panel)
        hbox.append(hippo.CanvasWidget(widget=self.grid), hippo.PACK_EXPAND)
        
        vbox.append(hbox, hippo.PACK_EXPAND)
        vbox.append(self.info_panel, hippo.PACK_END)

        canvas = hippo.Canvas()
        canvas.set_root(vbox)
        self.set_canvas(canvas)
        self.show_all()

        toolbox = ActivityToolbox(self)
        self.set_toolbox(toolbox)
        toolbox.show()

        self.pservice = presenceservice.get_instance()
        owner = self.pservice.get_owner()
        self.owner = owner

        # This displays the buddies_panel even if we fail to connect:
        self.buddies_panel.add_watcher(owner)
        self.info_panel.show(_('To play, share or invite someone.'))

        self.initiating = None

        self.game = None

        self.connect('shared', self._shared_cb)

        if self.shared_activity:
            # we are joining the activity
            self.buddies_panel.add_watcher(owner)
            self.connect('joined', self._joined_cb)
            self.shared_activity.connect('buddy-joined', 
                                          self._buddy_joined_cb)
            self.shared_activity.connect('buddy-left',
                                          self._buddy_left_cb)
            if self.get_shared():
                # oh, OK, we've already joined
                self._joined_cb()
        else:
            # we are creating the activity
            self.buddies_panel.remove_watcher(owner)
            self.buddies_panel.add_player(owner)
            #self.buddies_panel.set_is_playing(owner)
            #self.buddies_panel.set_count(owner, 69)

        self.connect('key-press-event', self.key_press_cb)

    def key_press_cb(self, widget, event):
        logger.debug('Keypress: %r, %r', widget, event)

        if self.game is not None:
            self.game.key_press_event(widget, event)

    def _shared_cb(self, activity):
        logger.debug('My Connect activity was shared')
        self.initiating = True
        self._setup()

        for buddy in self.shared_activity.get_joined_buddies():
            self.buddies_panel.add_watcher(buddy)

        self.shared_activity.connect('buddy-joined', self._buddy_joined_cb)
        self.shared_activity.connect('buddy-left', self._buddy_left_cb)

        logger.debug('This is my activity: making a tube...')
        id = self.tubes_chan[telepathy.CHANNEL_TYPE_TUBES].OfferDBusTube(
            SERVICE, {})
        self.info_panel.show(_('Waiting for another player to join.'))

    def _setup(self):
        if self.shared_activity is None:
            logger.error('Failed to share or join activity')
            return

        self.conn = self.shared_activity.telepathy_conn
        self.tubes_chan = self.shared_activity.telepathy_tubes_chan
        self.text_chan = self.shared_activity.telepathy_text_chan

        self.tubes_chan[telepathy.CHANNEL_TYPE_TUBES].connect_to_signal(
                'NewTube', self._new_tube_cb)

    def _list_tubes_reply_cb(self, tubes):
        for tube_info in tubes:
            self._new_tube_cb(*tube_info)

    def _list_tubes_error_cb(self, e):
        logger.error('ListTubes() failed: %s', e)

    def _joined_cb(self, activity):
        if self.game is not None:
            return

        if not self.shared_activity:
            return

        for buddy in self.shared_activity.get_joined_buddies():
            self.buddies_panel.add_watcher(buddy)

        logger.debug('Joined an existing Connect game')
        self.info_panel.show(_('Joined a game. Waiting for my turn...'))
        self.initiating = False
        self._setup()

        logger.debug('This is not my activity: waiting for a tube...')
        self.tubes_chan[telepathy.CHANNEL_TYPE_TUBES].ListTubes(
            reply_handler=self._list_tubes_reply_cb,
            error_handler=self._list_tubes_error_cb)

    def _new_tube_cb(self, id, initiator, type, service, params, state):
        logger.debug('New tube: ID=%d initator=%d type=%d service=%s '
                     'params=%r state=%d', id, initiator, type, service,
                     params, state)

        if (self.game is None and type == telepathy.TUBE_TYPE_DBUS and
            service == SERVICE):
            if state == telepathy.TUBE_STATE_LOCAL_PENDING:
                self.tubes_chan[
                    telepathy.CHANNEL_TYPE_TUBES].AcceptDBusTube(id)

            tube_conn = SugarTubeConnection(self.conn,
                self.tubes_chan[telepathy.CHANNEL_TYPE_TUBES],
                id, group_iface=self.text_chan[
                    telepathy.CHANNEL_INTERFACE_GROUP])
            self.game = ConnectGame(tube_conn, self.grid, self.initiating,
                self.buddies_panel, self.info_panel, self.owner, self)

    def _buddy_joined_cb (self, activity, buddy):
        logger.debug("buddy joined")
        self.buddies_panel.add_watcher(buddy)

    def _buddy_left_cb (self,  activity, buddy):
        logger.debug("buddy left")
        self.buddies_panel.remove_watcher(buddy)
